% script to load OFEN/BAFU data from LeChenit, Frontière, plot it and
% compute yearly maxima. 
%
% You can run this plot from another script by using:
% run('compute_yearly_maxima')

% a few settings
input_fname = '2371_Abfluss_Tagesmittel_1971-01-01_2022-11-30.csv';
output_fname = 'annual_maxima_LeChenit.csv';
year_sel = 2015;
flag_show_figure = true;
flag_export_data = true;

% import the daily data
if verLessThan('matlab','2020a') %code to run in MATLAB R2019b and earlier here 
    S = readtable(input_fname,'Delimiter',';','HeaderLines',8);
else %code to run in MATLAB R2020a and later here --
    S = readtable(input_fname,'Delimiter',';','NumHeaderLines',8);
end
S.Properties.VariableNames([7,9]) = {'date','flow'}; %rename from german

% create a logical index with data belonging to the selected year
q = year(S.date) == year_sel;

% prepare a square to be added to the plot (fill object)
date_short = S.date(q);
flow_short = S.flow(q);
X = [date_short(1),date_short(end),date_short(end),date_short(1)];
Y = [min(flow_short),min(flow_short),1.1*max(flow_short),1.1*max(flow_short)];

% make the figure
if flag_show_figure == 1
    
    figure
    subplot(2,2,1)
    hold on
    title('Daily streamflow')
    plot(S.date,S.flow)
    fill(X,Y,NaN,'EdgeColor','r')
    ylabel('m3/s')

    subplot(2,2,3)
    hold on
    title('Daily streamflows (log scale)')
    plot(S.date,S.flow)
    fill(X,Y,NaN,'EdgeColor','r')
    ylabel('m3/s')
    set(gca,'YScale','log') %to better see the low flows
    
    subplot(2,2,2)
    hold on
    title(sprintf('Daily streamflow in %d',year_sel))
    plot(S.date(q),S.flow(q))
    fill(X,Y,NaN,'EdgeColor','r')
    ylabel('m3/s')

    subplot(2,2,4)
    hold on
    title(sprintf('Daily streamflow in %d (log scale)',year_sel))
    plot(S.date(q),S.flow(q))
    fill(X,Y,NaN,'EdgeColor','r')
    ylabel('m3/s')
    set(gca,'YScale','log') %to better see the low flows

end

% compute the annual maxima
yrs = year(S.date);
unique_yrs = unique(yrs);
qmax = zeros(length(unique_yrs),1);
for i = 1:length(unique_yrs)
    qmax(i) = max(S.flow(year(S.date) == unique_yrs(i)));
end

% save to a csv file
if flag_export_data
    
    % create some header information on top
    fid = fopen(output_fname,'w+');
    fprintf(fid,['# This table includes yearly statistics of mean daily ',...
        'discharges [m3/s] computed for the BAFU/OFEV station: Orbe - Le Chenit, Frontière 2371\n' ...
        '# Original daily data is available in file %s, provided by BAFU/OFEV\n '],input_fname);
    fclose(fid);

    % append the data
    S2 = array2table([unique_yrs,qmax],'VariableNames',{'year','maxannualQ'});
    writetable(S2,output_fname,'WriteMode','Append','WriteVariableNames',true);
end

