
#include <webots/robot.h> 			// WEBOTS SPECIFIC
#include <webots/position_sensor.h> // WEBOTS SPECIFIC
#include <webots/distance_sensor.h>// WEBOTS SPECIFIC
#include <math.h>
#include <webots/motor.h> // WEBOTS SPECIFIC
#include "odometry.h"
#include "util.h"
  
  
WbDeviceTag ps[NB_SENSORS]; // handler for the distance sensors
static double ds_value[NB_SENSORS]; // array for distance measurements


/* Based on distance sensor values (stored in ds_value[*])
 * This function:
   - detects proximity to a wall
   - calculates the robot's position with the help of prior knowledge 
     and the current odometry estimate
   - Updates position estimate through the 'feature-based measure'
 */
void feature_update(pose_t* my_pose, double ds_value[NB_SENSORS]){
	bool wall_detected = false;
	double distance = 0;
	
	// TODO: Implement detection of front wall
	int wall= 130;
	if(ds_value[0] > wall || ds_value[7] > wall ){
		wall_detected = true;
	}
	
	if (!wall_detected){ // if no wall is detected we cannot update our position
		return;
	}
	
	/// TODO: reset the robot's X pose to the known value
	my_pose->x=1.0;
	printf("wall detected + reset\n");
}

void odometry_step(pose_t* my_pos){
  static double enc_left_prev,enc_right_prev;
  static double enc_left=0, enc_right=0;

	// update last enc values
	enc_left_prev = enc_left;
	enc_right_prev = enc_right;
	
	//// odometry
	// read encoders
	enc_left = wb_position_sensor_get_value(left_encoder);
	enc_right = wb_position_sensor_get_value(right_encoder);

	// calculate odometry update
	odo_compute_encoders(my_pos,enc_left-enc_left_prev,enc_right-enc_right_prev);
	
	// improve odometry through features
	feature_update(my_pos, ds_value);
}

void go(pose_t* my_pos, double target_x){

	  // define braitenberg coefficients: (const, we wont change this!), 
	  // note we ignore forward sensor!
	  const double braitenberg_coefficients[8][2] =
		{ {0, -0}, {110, -15}, {80, -10}, {-10, -10},
		{-15, -10}, {-5, 80}, {-30, 90}, {-0, 0} };
		
	  // variable declaration
	  double speed[2]; // variable to hold the calculated left/right speeds
	  double msl_w, msr_w; // variables to hold the wheel speeds
		
	// Repeat until 200 mm have been travelled
	while(fabs(target_x-my_pos->x)>=(double)0.01){
		
		// read sensor values
		for (int i = 0; i < NB_SENSORS; i++) 
		  ds_value[i] = wb_distance_sensor_get_value(ps[i]); // range: 0 (far) to 3800 (close)

	  
	 // nested loops computing the speed of each wheel
		for (int i = 0; i < 2; i++) {
		  speed[i] = 0.0;
		  for (int j = 0; j < 8; j++)
			speed[i] += braitenberg_coefficients[j][i] * (1.0 - (ds_value[j] / RANGE));
		  if(speed[i] > MAXIMUM_SPEED)
			speed[i] = MAXIMUM_SPEED;
		}
		
		// Set speed
		msl_w = speed[0]*MAX_SPEED_WEB/1000;
		msr_w = speed[1]*MAX_SPEED_WEB/1000;
		// check if motors have been initialized
		THROW_ERR(left_motor == 0, "util.c: left motor not initialized");
		THROW_ERR(right_motor == 0, "util.c: right motor not initialized");
		wb_motor_set_velocity(left_motor, msl_w);
		wb_motor_set_velocity(right_motor, msr_w);
			
		odometry_step(my_pos);	// Get estimated Dead Reckoning
		
		
		// run simulation step
		wb_robot_step(TIME_STEP);
	}
	
}

void turn(pose_t* my_pos, double angle){
	double msl_w, msr_w; // variables to hold the wheel speeds
		
	// Repeat until angle rad have been turned
	while(fabs(angle-my_pos->heading)>=0.1){
		// Set speed
		msl_w = -200*MAX_SPEED_WEB/1000;
		msr_w = 200*MAX_SPEED_WEB/1000;
		
		// check if motors have been initialized
		THROW_ERR(left_motor == 0, "util.c: left motor not initialized");
		THROW_ERR(right_motor == 0, "util.c: right motor not initialized");
		wb_motor_set_velocity(left_motor, msl_w);
		wb_motor_set_velocity(right_motor, msr_w);
			
		odometry_step(my_pos);	// Get estimated Dead Reckoning
		my_pos->heading = fmod(my_pos->heading,6.28318530718);
		
		// run simulation step
		wb_robot_step(TIME_STEP);
	}
	
}

int main(int argc, const char *argv[]) {
  char name[] = "ps0"; 
    
  // initiliaze webots
  wb_robot_init();
  // initialize the robot's motors + sensors
  init_robot_wb();
	    
  // get and enable each distance sensor
  for(int i = 0; i < NB_SENSORS; i++) {
    ps[i] = wb_robot_get_device(name);
    wb_distance_sensor_enable(ps[i], TIME_STEP);
    name[2]++; // increase the device name to "ps1", "ps2", etc.
  }
  
  // initialize my position
  pose_t my_pos = {0,0,0};
  
  // reset odometry
  odo_reset(TIME_STEP);
  
  // main loop
  while(wb_robot_step(TIME_STEP)!=-1) {
	  for(int i=0;i<3;i++){		  
		go(&my_pos,1.0);
		turn(&my_pos,3.1415);  
		printf("pose %f %f %f\n",my_pos.x,my_pos.y,my_pos.heading);
		go(&my_pos,0);
		turn(&my_pos,0);
	  }
  }
	// stop the robot
    wb_motor_set_velocity(left_motor, 0);
    wb_motor_set_velocity(right_motor, 0);
	
	wb_robot_cleanup();
  
 
  return 0;
}