#include "util.h"
#include <webots/motor.h>

/* Initializes the motors in WB
 */
void init_robot_wb(){
  //get motors
  left_motor = wb_robot_get_device("left wheel motor");
  right_motor = wb_robot_get_device("right wheel motor");
  // check they exist
  THROW_ERR(left_motor == 0, "util.c: left motor could not be initialized");
  THROW_ERR(right_motor == 0, "util.c: right motor could not be initialized");
  // set to velocity control
  wb_motor_set_position(left_motor, INFINITY);
  wb_motor_set_position(right_motor, INFINITY);
  // set speeds to 0  
  wb_motor_set_velocity(left_motor, 0);
  wb_motor_set_velocity(right_motor, 0);
  
  // get encoders
  left_encoder = wb_robot_get_device("left wheel sensor");
  right_encoder = wb_robot_get_device("right wheel sensor");
  // check they exist
  THROW_ERR(left_encoder == 0, "util.c: left motor encoder could not be initialized");
  THROW_ERR(right_encoder == 0, "util.c: right motor encoder could not be initialized");
  // get timestep of webots
  int timestep = wb_robot_get_basic_time_step();
  // verify it's set
  THROW_ERR(timestep ==0, "util.c: Timestep not set!");
  // enable them
  wb_position_sensor_enable(left_encoder, timestep);
  wb_position_sensor_enable(right_encoder, timestep);
  // run simulation for basic timestep to finish initialization
  wb_robot_step(timestep);
}


/* calculates and sets motor speeds for braitenberg
 * Input: current distance sensor values
 * Remark: 'const' indicates that this function will not change the values of ds_value
 */
void braitenberg_step(const double ds_value[NB_SENSORS]){ 
  // define braitenberg coefficients: (const again, we wont change this!)
  const double braitenberg_coefficients[8][2] =
    { {140, -35}, {110, -15}, {80, -10}, {-10, -10},
    {-15, -10}, {-5, 80}, {-30, 90}, {-20, 160} };
	
  // variable declaration
  int i,j; // loop variables
  double speed[2]; // variable to hold the calculated left/right speeds
  double msl_w, msr_w; // variables to hold the wheel speeds
  
 // nested loops computing the speed of each wheel
    for (i = 0; i < 2; i++) {
      speed[i] = 0.0;
      for (j = 0; j < 8; j++)
        speed[i] += braitenberg_coefficients[j][i] * (1.0 - (ds_value[j] / RANGE));
      if(speed[i] > MAXIMUM_SPEED)
        speed[i] = MAXIMUM_SPEED;
    }
    
    // Set speed
    msl_w = speed[0]*MAX_SPEED_WEB/1000;
    msr_w = speed[1]*MAX_SPEED_WEB/1000;
	// check if motors have been initialized
	THROW_ERR(left_motor == 0, "util.c: left motor not initialized");
	THROW_ERR(right_motor == 0, "util.c: right motor not initialized");
    wb_motor_set_velocity(left_motor, msl_w);
    wb_motor_set_velocity(right_motor, msr_w);
}


/* throws an error and aborts the program
 *
 */
void throw_error(bool test, char* message){
	if(test){
		fprintf(stderr,message,"string format", 30);
		fprintf(stderr,"\n");
		exit(false);
	}
}


double IR_to_distance(double proximity){
    /* Uses values defined in webots proto
          0 4095 0.005
          0.005 3474 0.037
          0.01 2211 0.071
          0.02 676 0.105
          0.03 306 0.125
          0.04 164 0.206
          0.05 90 0.269
          0.06 56 0.438
          0.07 34 0.704
	*/

    double prox_values [9] = {4095, 3474, 2211, 676, 306, 164, 90, 56, 34};
    double dist_values [9] = {0, 0.005, 0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07};

    if (proximity > 4094) {
        return 0;
    }
	
    int a_index, b_index;
    double x_a , x_b ;
    double y_a , y_b ;
    for(int i=0; i<9-1; i++){
        if (proximity >= prox_values[i + 1]) {
            // linear interpolations
            a_index = i;
            b_index = i+1;
            x_a = prox_values[a_index];
            x_b = prox_values[b_index];
            y_a = dist_values[a_index];
            y_b = dist_values[b_index];
            return y_a + (proximity - x_a)*(y_b-y_a)/(x_b-x_a);
        }
    }
    return -1;
}








///////////////// MATRIX UTILITIES ///////////////////

void print(int N, int M, double mat[N][M]){
	for (int i=0;i<N;i++){
		for (int j=0;j<M;j++){
			printf("%f ",mat[i][j]);
		}
		printf("\n");
	}
}



// Multiplies two matrices mat1[][] and mat2[][]
// and returns result.
// (m1) x (m2) and (n1) x (n2) are dimensions
// of given matrices.
void multiply(int m1, int m2, const double mat1[][m2], int n1,
              int n2, const double mat2[][n2], double res[m1][n2])
{
    int x, i, j;
    for (i = 0; i < m1; i++) 
    {
        for (j = 0; j < n2; j++) 
        {
            res[i][j] = 0;
            for (x = 0; x < m2; x++) 
            {
                *(*(res + i) + j) += *(*(mat1 + i) + x)
                                     * *(*(mat2 + x) + j);
            }
        }
    }
}


void inverse(int matsize,double I[matsize][matsize]){
	
	double A[matsize][matsize];
	int i,j,k;
	for(i=0;i<matsize;i++){
		for(j=0;j<matsize;j++){
			A[i][j] = I[i][j];
			I[i][j] = (i==j ? 1 : 0);
		}
	}
	
    for(k=0;k<matsize;k++)                                  
    {      
        double temp=A[k][k];                   //'temp'  
        // stores the A[k][k] value so that A[k][k]  will not change
        for(j=0;j<matsize;j++)      //during the operation //A[i] //[j]/=A[k][k]  when i=j=k
        {
            A[k][j]/=temp;                                  //it performs // the following row operations to make A to unit matrix
            I[k][j]/=temp;                                  //R0=R0/A[0][0],similarly for I also R0=R0/A[0][0]
        }                                                   //R1=R1-R0*A[1][0] similarly for I
        for(i=0;i<matsize;i++)                              //R2=R2-R0*A[2][0]      ,,
        {
            temp=A[i][k];                       //R1=R1/A[1][1]
            for(j=0;j<matsize;j++)             //R0=R0-R1*A[0][1]
            {                                   //R2=R2-R1*A[2][1]
                if(i==k)
                    break;                      //R2=R2/A[2][2]
                A[i][j]-=A[k][j]*temp;          //R0=R0-R2*A[0][2]
                I[i][j]-=I[k][j]*temp;          //R1=R1-R2*A[1][2]
            }
        }
    }
}

// Adds two matrices mat1[][] and mat2[][] and returns result.
// (m1) x (m2) are dimensions of given matrices.
void add(int m1, int m2, const double mat1[m1][m2], const double mat2[m1][m2], double res[m1][m2])
{
    int i, j;
    for (i = 0; i < m1; i++) 
    {
        for (j = 0; j < m2; j++) 
        {
            res[i][j] = mat1[i][j] + mat2[i][j];
        }
    }
}

// Substracts matrice mat2[][] from mat1[][] and returns result.
// (m1) x (m2) are dimensions of given matrices.
void substract(int m1, int m2, const double mat1[m1][m2], const double mat2[m1][m2], double res[m1][m2])
{
    int i, j;
    for (i = 0; i < m1; i++) 
    {
        for (j = 0; j < m2; j++) 
        {
            res[i][j] = mat1[i][j] - mat2[i][j];
        }
    }
}

