# -*- coding: utf-8 -*-
"""
Méthodes d'estimation
SIE 3e semestre, GC 5e semestre

ESO-TOPO / 03.12.24

--------------------------------------------------------------------------

Par rapport au code initial, non seulement les trous sont comblés, mais
le paramètre "w = vitesse angulaire omega" en radians est remplacé par le
paramètre "T = période" en heures, nettement plus facile à interpréter,
notamment pour tester si la période est significativement différente de
24 heures.

En avant-première, on introduit une friandise du chapitre 5 "Fiabilité":
le résidu standardisé (formule 5.12, p.114/2021), souvent confondu avec
le quotient local (p.56/2021).
"""

import numpy as np
import matplotlib.pyplot as plt
np.set_printoptions(precision=4, suppress=True)

print("\n","Ajustement d'une sinusoïde")

# Données

t = np.array([-12.4, -4.2, 1.2, 6.3, 12.1, 18.9, 24.2, 31.6])
y = np.array([5.8, 5.2, 11.4, 12.5, 5.3, 4.4, 10.7, 10.8])
n = len(y)
y = y.reshape((-1,1)) # vecteur-colonne pour les observations

# MODELE STOCHASTIQUE

sigma0pri = 0.2
Ql = np.identity(n)
P = np.linalg.inv(Ql)

# MODELE FONCTIONNEL

cas = 1
#cas = int(input(' indice du modèle choisi (1 à 4) : '))
print(' cas = ', cas)

# Pour chaque paramètre, on définit une valeur approchée et la résolution
# souhaitée de la valeur compensée. Cette résolution est utilisée:
# - comme incrément pour la linéarisation numérique;
# - comme seuil pour le test de convergence.
# attention aux unités choisies!

if cas == 1:
    cbulle = 8.0
    xbulle = np.array([cbulle])
    xbulle = xbulle.reshape((-1,1)) # vecteur-colonne pour les paramètres
    print('\n','xbulle =\n',xbulle)
    inc = np.array([0.1]) # incrément pour c  

elif cas == 2:
    cbulle = 8.0
    abulle = 5.0
    Tbulle = 24.0 # h
    xbulle = np.array([cbulle, abulle, Tbulle])
    xbulle = xbulle.reshape((-1,1)) # vecteur-colonne pour les paramètres
    print('\n','xbulle =\n',xbulle)
    inc = np.array([0.1, 0.1, 1/60]) # pour c, a et T (1 min)

elif cas == 3:
    cbulle = 7.9443
    abulle = 5.0578
    Tbulle = 24.0    # h
    fibulle = 0.5118 # rad (2*np.pi = 24 h)
    xbulle = np.array([cbulle, abulle, Tbulle, fibulle])
    xbulle = xbulle.reshape((-1,1)) # vecteur-colonne pour les paramètres
    print('\n','xbulle =\n',xbulle)
    inc = np.array([0.1, 0.1, 1/60, np.pi/720]) # pour c, a, T et fi (1 min)

elif cas == 4:
    cbulle = 8.0
    abulle = 5.0
    fibulle = np.pi/6 # rad
    xbulle = np.array([cbulle, abulle, fibulle])
    xbulle = xbulle.reshape((-1,1)) # vecteur-colonne pour les paramètres
    print('\n','xbulle =\n',xbulle)
    inc = np.array([0.1, 0.1, np.pi/720]) # pour c, a et fi

else:

    print('\n','ERREUR: pas de modèle avec cet indice!\n')

u = len(xbulle)

"""
COMPENSATION itérative

D'abord on reporte le modèle approché, car les paramètres approchés
seront remplacés au cours des itérations.

Pour éviter que le graphe de la fonction soit une ligne brisée,
on calcule sa valeur de façon plus dense que les observations.
On définit tt comme une série allant de t(1)-marge à t(n)+marge,
avec un petit incrément.
"""

from sinusoids import sinusoids

tt = np.arange(t[0]-1, t[n-1]+1, 0.1)
nt = len(tt)

ybullet = np.zeros(nt)
for i in range(nt):
    ybullet[i] = sinusoids(cas, tt[i], xbulle)

plt.plot(tt,ybullet,'g:',label='fonction approchée')
plt.xlabel('t')
plt.ylabel('y')
plt.axis([-16, 36, 2, 14])
plt.xticks(np.arange(-18, 36, 6))
plt.yticks(np.arange(2, 14, 2))

"""
La boucle "while" est stoppée si:
- chaque incrément est inférieur au seuil défini
  (convergence, peut-être vers la bonne solution!);
- le nombre d'itérations atteint "maxit"
  (divergence, là c'est carrément l'échec!).
"""

it = -1                    # initialisation du nombre d'itérations
dx = np.ones(len(xbulle))  # incrément fictif pour exiger le premier calcul 
maxit = 10                 # nombre maximal d'itérations

ybulle = np.zeros((n,1))
vbulle = np.zeros((n,1))
A = np.zeros((n,u))
while np.amax(abs(dx)/inc) > 1 and it < maxit :

    it += 1

    # observations et résidus approchés
    for i in range(n):
        ybulle[i,0] = sinusoids(cas,t[i],xbulle)
        vbulle[i,0] = y[i,0] - ybulle[i,0]

        # Construction de A par différentiation numérique
        # Ce procédé est valable pour toute fonction (dérivable) si les incréments
        # sont adéquats. Il est utile surtout pour une fonction compliquée.
        for j in range(u): 
            xinc = xbulle.copy()
            xinc[j,0] = (xbulle[j,0] + inc[j])
            A[i,j] = (sinusoids(cas,t[i],xinc) - ybulle[i,0])/inc[j]

    # cofacteurs, incréments et paramètres compensés
    Qxcomp = np.linalg.inv(A.T @ P @ A)  
    dx = Qxcomp @ A.T @ P @ vbulle
    print('\n','dx =\n',dx) # affichage utile pendant l'élaboration du code
    xcomp = xbulle + dx

    # paramètres approchés pour la prochaine itération
    xbulle = xcomp.copy()

print('\n','it = ',it)  # impression du nombre d'itérations

# observations et résidus compensés

ycomp = np.zeros((n,1))
vcomp = np.zeros((n,1))
for i in range(n):
    ycomp[i,0] = sinusoids(cas,t[i],xcomp)
    vcomp[i,0] = y[i,0] - ycomp[i,0]

# impression des résultats

print('\n','xcomp =\n',xcomp)
print('\n','ycomp =\n',ycomp)
print('\n','vcomp =\n',vcomp)

# Sur le graphique, on ajoute le modèle compensé.

ycompt = np.zeros(len(tt))
for i in range(nt):
    ycompt[i]  = sinusoids(cas,tt[i],xcomp)

plt.plot(tt,ycompt,'r-',label='fonction compensée')

# Finalement on reporte les observations. Ainsi elles ne sont pas couvertes
# par les modèles.

plt.plot(t,y,'bo',label='observations')

plt.legend()
plt.show()

# -------------------------------------
# ANALYSE DES RESULTATS

# analyse globale: adéquation des mesures et du modèle

sigma0pos = np.sqrt(vcomp.T @ P @ vcomp/(n-u))
print('\n','sigma0pos = %.2f' % sigma0pos)
ratioglo = sigma0pos/sigma0pri   # quotient global
print('\n','ratioglo = %.2f' % ratioglo)

# analyse locale: détection de faute

# On calcule les "quotients locaux" en divisant chaque résidu vcomp(i) par
# l'écart-type sigmal(i) de l'observation correspondante, voir p.56 (2021).
# Un quotient supérieur à 2 ou 3 est suspect. Ce test est simple, mais pas
# vraiment rigoureux du point de vue statistique, car on compare une valeur
# à l'écart-type d'une autre.

# En l'occurrence, puisque la matrice des cofacteurs Ql est l'identité,
# chaque élément du vecteur sigmal est égal à sigma0pri.

ratioloc = vcomp/sigma0pri   # quotients locaux
print('\n','ratioloc =\n',ratioloc)

# cofacteurs des observations compensées et des résidus (compensés)

Qlcomp = A @ Qxcomp @ A.T
Qvcomp = Ql - Qlcomp

# écarts-types et corrélations des résidus compensés

from covmat2cormat import covmat2cormat

[qv,correl_vcomp] = covmat2cormat(Qvcomp)
sigmav = sigma0pos*qv.T   # cofacteurs multipliés par sigma0 (pri ou pos)
sigmav = sigmav.reshape((-1,1)) # vecteur-colonne
print('\n','sigmav =\n',sigmav)

"""
On calcule les "résidus standardisés" en divisant chaque résidu vcomp(i)
par son propre écart-type sigmav(i), voir formule 5.12, p.114/2021.
Une valeur supérieure à 2 (95%) ou 3 (99%) est suspecte.
Pour un test statistique, cette valeur est plus logique que le quotient local.
Toutefois, une mesure qui n'est pas contrôlée par les autres n'est pas
influencée par la compensation. Son résidu compensé est nul et son résidu
standardisé tend vers 0/0. Autrement dit, il est indéterminé.
Numériquement il devient instable.
"""

vstandard = vcomp/sigmav   # résidus standardisés
print('\n','vstandard =\n',vstandard)

np.set_printoptions(precision=2, suppress=True)
print('\n','correl_vcomp =\n',correl_vcomp)

# écarts-types et corrélations des paramètres compensés

[qx,correl_xcomp] = covmat2cormat(Qxcomp)
sigmax = sigma0pos*qx   # cofacteurs à multiplier par sigma0 (pri ou pos)
sigmax = sigmax.reshape((-1,1)) # vecteur-colonne
np.set_printoptions(precision=4, suppress=True)
print('\n','sigmax =\n',sigmax)

np.set_printoptions(precision=2, suppress=True)
print('\n','correl_xcomp =\n',correl_xcomp)

# Période estimée significativement différente de 24 heures?
# C'est probable pour un quotient supérieur à 2 (95%) ou 3 (99%).

if cas == 3:
    Tcomp = xcomp[2,0]
    sigmaTcomp = sigmax[2,0]
    ratioTcomp = (Tcomp-24)/sigmaTcomp

    print('\n','Tcomp = %.4f' % Tcomp)
    print('\n','sigmaTcomp = %.4f' % sigmaTcomp)
    print('\n','ratioTcomp = %.2f' % ratioTcomp)


"""
FIABILITE INTERNE 
D'abord on calcule le parte de redondance selon (5.6)
et on vérifie si sum(z)= r
Après on calcule le plus petite faut detectable (5.14)
"""
[
ql,correl_lcomp] = covmat2cormat(Qlcomp)
sigmalcomp = sigma0pri*ql.T   # cofacteurs multipliés par sigma0 (pri ou pos)
sigmalcomp = sigmalcomp.reshape((-1,1)) # vecteur-colonne
print('\n','sigmalcomp =\n',sigmav)

z     = np.zeros((n,1)) # part de redondance
nabl1 = np.zeros((n,1)) # nabla, faut detectable via sigma_v 
#nabl2 = np.zeros((n,1)) # idem pour la controle, depend sigma_pri
mnabl = np.zeros((n,n)) # matrice pour la fiabilite externe 
for i in range(n):
    z[i] = Qvcomp[i,i]*P[i,i]
    nabl1[i] = sigmav[i]/z[i] * 4.1 
   # nabl2[i] = sigmalcomp[i]/z[i] * 4.1
    mnabl[i,i] = nabl1[i]

print('\n','zi =\n',z) 
print('\n','r = sum(z_i) =',sum(z))
print('\n','nabl1 =\n',nabl1)

"""
ANALYSE INTERNE  
 cas 3   variation plus grand (valeurs 1.3 - 1.8) sensitivite au 1er et dernière mesures
 cas 4   max faut non-detéctable est plus petite (~1.2), pas grande variation entre mesures

"""

"""
FIABILITE EXTERNE 
matrice n x u selon 5.18 
"""
mnablax = Qxcomp @ A.T @ P @ mnabl

np.set_printoptions(precision=2, suppress=True)
print('\n','mnabl =\n',mnablax.T)

"""
ANALYSE EXTERNE 
cas3   - amplitude - max sensitivite 0.4, non homogene dans le temps 
       - phase  - max sens. 0.11 
 
cas 4  - amplitude - max sesitivité 2x fois plus petite (<0.2) mois de varations
       - phase - 2x mois sensible que dans cas 3 (<0.05), plus homogène dans le temps
"""
