package com.epfl.esl.sportstracker

import android.app.Activity
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.compose.setContent
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import coil.compose.AsyncImage
import com.epfl.esl.sportstracker.ui.theme.SportsTrackerTheme

class MainActivity : ComponentActivity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        setContent {
            SportsTrackerTheme {
                HomeScreen()
            }
        }
    }
}

@Composable
fun HomeScreen(modifier: Modifier = Modifier) {
    Surface(
        modifier = modifier.fillMaxSize(),
        color = MaterialTheme.colorScheme.background
    ) {
        var username by remember { mutableStateOf("") }
        var password by remember { mutableStateOf("") }
        var isEditingMode by remember { mutableStateOf(true) }
        var imageUri by remember { mutableStateOf<Uri?>(null) }
        var resultLauncher = rememberLauncherForActivityResult(
            contract = ActivityResultContracts.StartActivityForResult(),
            onResult = { result ->
                if (result.resultCode == Activity.RESULT_OK) {
                    val uri = result.data?.data
                    imageUri = uri
                }
            }
        )

        if (isEditingMode) {
            HomeContentEditing(
                imageUri = imageUri,
                username = username,
                password = password,
                onUsernameChanged = { newValue -> username = newValue },
                onPasswordChanged = { newValue -> password = newValue },
                onContinueButtonClicked = { isEditingMode = false },
                onPickImageButtonClicked = {
                    val intent = Intent(Intent.ACTION_GET_CONTENT)
                    intent.type = "image/*"
                    resultLauncher.launch(intent)
                },
                modifier
            )
        } else {
            HomeContentDisplaying(
                imageUri = imageUri,
                username = username,
                onUpdateButtonClicked = {
                    isEditingMode = true
                },
                onLogOutButtonClicked = {
                    username = ""
                    password = ""
                    imageUri = null
                    isEditingMode = true
                })
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun HomeContentEditing(
    imageUri: Uri?,
    username: String,
    password: String,
    onUsernameChanged: (String) -> Unit,
    onPasswordChanged: (String) -> Unit,
    onContinueButtonClicked: () -> Unit,
    onPickImageButtonClicked: () -> Unit,
    modifier: Modifier = Modifier
) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        if (imageUri == null) {
            Image(
                painter = painterResource(id = R.drawable.user_image),
                contentDescription = stringResource(R.string.default_user_image),
                modifier = modifier
                    .fillMaxWidth()
                    .height(500.dp)
            )
        } else {
            AsyncImage(
                model = imageUri,
                contentDescription = stringResource(R.string.picked_user_image),
                modifier = modifier
                    .fillMaxWidth()
                    .height(500.dp)
            )
        }
        TextField(
            value = username,
            onValueChange = onUsernameChanged,
            label = {
                Text(stringResource(R.string.username_hint))
            },
            textStyle = TextStyle(fontSize = 24.sp, textAlign = TextAlign.Center),
            singleLine = true,
            modifier = modifier
                .fillMaxWidth()
                .padding(bottom = 8.dp)
        )
        TextField(
            value = password,
            onValueChange = onPasswordChanged,
            label = {
                Text(stringResource(R.string.password_hint))
            },
            textStyle = TextStyle(fontSize = 24.sp, textAlign = TextAlign.Center),
            singleLine = true,
            visualTransformation = PasswordVisualTransformation(),
            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Password),
            modifier = modifier.fillMaxWidth()
        )
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(top = 16.dp)
        ) {
            Button(
                onClick = onContinueButtonClicked,
                modifier = Modifier.weight(1f)
            ) {
                Text(text = stringResource(R.string.confirm_button_text))
            }
            Button(
                onClick = onPickImageButtonClicked,
                modifier = Modifier.weight(1f)
            ) {
                Text(text = stringResource(R.string.pick_image_button_text))
            }
        }
    }
}

@Composable
fun HomeContentDisplaying(
    imageUri: Uri?,
    username: String,
    onUpdateButtonClicked: () -> Unit,
    onLogOutButtonClicked: () -> Unit,
    modifier: Modifier = Modifier
) {
    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        if (imageUri == null) {
            Image(
                painter = painterResource(id = R.drawable.user_image),
                contentDescription = stringResource(R.string.default_user_image),
                modifier = modifier
                    .fillMaxWidth()
                    .height(500.dp)
            )
        } else {
            AsyncImage(
                model = imageUri,
                contentDescription = stringResource(R.string.picked_user_image),
                modifier = modifier
                    .fillMaxWidth()
                    .height(500.dp)
            )
        }
        Text(
            text = username,
            style = TextStyle(fontSize = 24.sp, textAlign = TextAlign.Center),
            maxLines = 1,
            modifier = modifier
                .fillMaxWidth()
                .padding(bottom = 8.dp)
        )
        Text(
            text = stringResource(R.string.password_hidden_text),
            style = TextStyle(fontSize = 24.sp, textAlign = TextAlign.Center),
            maxLines = 1,
            modifier = modifier.fillMaxWidth()
        )
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(top = 16.dp)
        ) {
            Button(
                onClick = onUpdateButtonClicked,
                modifier = Modifier.weight(1f)
            ) {
                Text(text = stringResource(R.string.update_button_text))
            }
            Button(
                onClick = onLogOutButtonClicked,
                modifier = Modifier.weight(1f)
            ) {
                Text(text = stringResource(R.string.log_out_button_text))
            }
        }
    }
}

@Preview
@Composable
fun HomeScreenPreview() {
    SportsTrackerTheme {
        HomeScreen()
    }
}

@Preview
@Composable
fun HomeScreenDisplayingPreview() {
    SportsTrackerTheme {
        HomeContentDisplaying(null, "username", {}, {})
    }

}

@Preview
@Composable
fun HomeScreenEditingPreview() {
    SportsTrackerTheme {
        HomeContentEditing(null, "", "", {}, {}, {}, {})
    }
}