from moteur_psc.contrainte import Contrainte

class ContrainteAxiomeCadre(Contrainte):
    """ Contrainte forçant l'utilisation d'opérateurs pour agir sur les\
        variables d'un problème."""

    def __init__(self, var_pre, ops, var_post):
        """
            :param var_pre: la variable au début de l'état - prop(Si).
            :param list var_ops: les variables correspondant aux opérateurs qui\
            ont prop comme postcondition.
            :param var_post: la variable à la fin de l'état - prop(Si+1).
        """

        Contrainte.__init__(self, (var_pre, var_post) + tuple(ops))

        self.var_pre = var_pre
        self.var_post = var_post
        self.vars_ops = ops

    def est_valide(self, var, val):
        """ Evalue la validité de la contrainte pour une valeur de variable\
            donnée.

            L'évaluation est paresseuse : si au moins une variable n'est pas\
            instanciée, on retourne ``True``.
            Sinon, si la variable change de valeur, alors au moins une variable\
            d'opérateur doit être ``True``.

            :param var: la variable à laquelle assigner la valeur ``val``.
            :param val: la valeur à assigner à ``var``.
            :return: ``True`` si la contrainte est valide pour la paire\
            variable/valeur passée en paramètre.
        """
        ancienne_valeur = var.val
        var.val = val

        # On part du principe que la contrainte est valide si au moins une
        # variable n'est pas instanciée.
        for var2 in self.variables:
            if var2.val is None:
                var.val = ancienne_valeur
                return True

        valide = False
        # Si toutes les variables sont instanciées et qu'une variable passe de
        # False à True.
        if self.var_pre.val == False and self.var_post.val == True:
            # Vérifie qu'au moins un des opérateurs est appliqué.
            for op in self.vars_ops:
                if op.val == True:
                    valide = True
                    break
        else:
            valide = True

        var.val = ancienne_valeur
        return valide

    def propage(self, var):
        """ Propage la nouvelle valeur d'une variable afin de vérifier que la\
            contrainte est toujours satisfaisable.

            La propagation est paresseuse: elle retourne ``True`` si plus d'une\
            variable n'a pas encore la valeur ``None``.

            :param var: la variable ayant reçu une nouvelle valeur.
            :return: ``True`` si la contrainte peut encore être satisfaite.
        """
        var2 = None
        for var_i in self.variables:
            if var_i.val is None:
                if var2 is None:
                    var2 = var_i
                else:
                    # Il reste plus d'une variable à instancier.
                    return True

        # Teste les valeurs du label pour la dernière variable non instanciée.
        for val in var2.label[:]:
            if not self.est_valide(var2, val):
                var2.label.remove(val)

        return len(var2.label) > 0

    def reviser(self):
        """ Réviser n'est pas définie pour les contraintes n-aires.

            :return: ``False``.
        """

        return False

    def __repr__(self):
        return 'Axiome de cadre:\n\t{}\n\t{}\n\t{}'.format(self.var_pre,
                                                           [op for op in self.vars_ops],
                                                           self.var_post)