from moteur_psc.contrainte import ContrainteUnaire
from moteur_psc_heuristique.contrainte_avec_propagation import ContrainteAvecPropagation
from moteur_psc_heuristique.psc_heuristique import PSCHeuristique
from moteur_psc_planification.axiomecadre import ContrainteAxiomeCadre
from .etat import Etat

class Planification:
    def __init__(self, propositions, operateurs, 
                 mutex_propositions, mutex_operateurs, 
                 depart, but, nb_etats):
        """
            :param list propositions: liste des propositions du problème.
            :param list operateurs: liste des opérateurs du problème.
            :param list mutex_propositions: liste des mutex de propositions.
            :param list mutex_operateurs: liste des mutex d'opérateurs.
            :param depart: contraintes de départ.
            :param but: contraintes finales.
            :param nb_etats: nombre d'états du plan.
        """

        self.operateurs = operateurs
        self.mutex_propositions = mutex_propositions
        self.mutex_operateurs = mutex_operateurs

        self.depart = depart
        self.but = but

        self.nb_etats = nb_etats

        self.propositions = propositions

        self.etats = []
        self.construire_etats()
        
        self.psc = PSCHeuristique(self.variables(), self.construire_contraintes())

    def construire_etats(self):
        """ Construction des états de la planification.

            Chaque état est constitué de ses variables initiales, de ses\
            variables d'opérateurs et de ses variables finales.
            
            /!\ Les variables finales d'un état coïncident avec les variables\
            initiales de l'état suivant /!\.
        """
        self.etats.append(Etat(0, self.propositions, self.operateurs, None))

        for i in range(1, self.nb_etats):
            self.etats.append(Etat(i, self.propositions, 
                                   self.operateurs, self.etats[-1]))

    def variables(self):
        """
            :return: la liste des variables du problème transformé en PSC.
        """

        # Utiliser un set évite les doublons entre variables finales et
        # initiales.
        variables = set()
        for etat in self.etats:
            variables.update(etat.variables())

        return list(variables)

    def construire_contraintes(self):
        """ Méthode intermédiaire pour faire l'aggrégation de toutes les\
            contraintes.

            :return: les contraintes du problème transformé en PSC.
        """

        return (self.construire_contraintes_propositions() +
                self.construire_contraintes_operateurs() +
                self.construire_contraintes_conditions() +
                self.construire_contraintes_axiomes_cadre() +
                self.construire_contraintes_initiales() +
                self.construire_contraintes_finales())

    def construire_contraintes_propositions(self):
        """ Construit les contraintes traduisant les mutex de propositions.

            :return: une liste de contraintes.
        """
        contraintes = []
        nand = lambda x,y: not (x and y)

        # Construction des contraintes générées par les mutex de propositions
        # **pour chaque état**.
        for mutex in self.mutex_propositions:
            for etat in self.etats:
                contr = ContrainteAvecPropagation(etat.vars_initiales[mutex[0]], 
                                                  etat.vars_initiales[mutex[1]], 
                                                  nand)
                contraintes.append(contr)
                # Les mutex de propositions doivent aussi être valides pour les
                # variables finales du dernier état.
                if etat.no_etat == (self.nb_etats - 1):
                    contr = ContrainteAvecPropagation(etat.vars_finales[mutex[0]], 
                                                      etat.vars_finales[mutex[1]], 
                                                      nand)
                    contraintes.append(contr)

        return contraintes

    def construire_contraintes_operateurs(self):
        """ Construit les contraintes traduisant les mutex d'opérateurs.

            :return: une liste de Contraintes.
        """
        contraintes = []
        nand = lambda x,y: not (x and y)

        for mutex in self.mutex_operateurs:
            for etat in self.etats:
                contr = ContrainteAvecPropagation(etat.vars_operateurs[mutex[0].nom], 
                                                  etat.vars_operateurs[mutex[1].nom], 
                                                  nand)
                contraintes.append(contr)

        return contraintes

    def construire_contraintes_conditions(self):
        """ Constuit les contraintes traduisant les pré- et post-conditions des\
            opérateurs.

            :return: une liste de contraintes.
        """
        contraintes = []
        # Contraintes générées par les pré- et post-conditions.
        # Implication logique.
        imp = lambda x,y: (not x) or y

        for etat in self.etats:
            for op in self.operateurs:
                for precond in op.precond:
                    contr = ContrainteAvecPropagation(etat.vars_operateurs[op.nom], 
                                                      etat.vars_initiales[precond], 
                                                      imp)
                    contraintes.append(contr)
                for postcond in op.postcond:
                    contr = ContrainteAvecPropagation(etat.vars_operateurs[op.nom], 
                                                      etat.vars_finales[postcond], 
                                                      imp)
                    contraintes.append(contr)

        return contraintes

    def construire_contraintes_axiomes_cadre(self):
        """ Construit les contraintes d'axiomes de cadre.

            Un axiome de cadre traduit le fait qu'une variable ne peut pas\
            changer de valeur sans l'action d'un opérateur.

            :return: une liste de contraintes.
        """
        contraintes = []

        for etat in self.etats:
            for prop in self.propositions:
                vars_ops = [etat.vars_operateurs[op.nom] 
                            for op in self.operateurs 
                            if prop in op.postcond]
                contr = ContrainteAxiomeCadre(etat.vars_initiales[prop], 
                                              vars_ops, 
                                              etat.vars_finales[prop])
                contraintes.append(contr)
        return contraintes

    def construire_contraintes_initiales(self):
        """ Construit les contraintes qui fixent la situation de départ.

            :return: une liste de contraintes.
        """
        contraintes = []
        for contrainte in self.depart:
            eq = lambda x: x == contrainte[1]
            contr = ContrainteUnaire(self.etats[0].vars_initiales[contrainte[0]], eq)
            contraintes.append(contr)

        return contraintes

    def construire_contraintes_finales(self):
        """ Construit les contraintes qui fixent la situation but.

            :return: une liste de contraintes.
        """
        contraintes = []

        for contrainte in self.but:
            eq = lambda x: x == contrainte[1]
            contr = ContrainteUnaire(self.etats[-1].vars_finales[contrainte[0]], eq)
            contraintes.append(contr)

        return contraintes

    def resoudre(self):
        """ Utilise le Forward Checking pour résoudre le problème de\
            planification transformé en PSC.

            Note: La méthode ``solution`` permet d'afficher la solution sans\
            avoir à traiter la valeur de retour de cette méthode.

            :return: le dictionnaire ``{variable : valeur}`` retourné par la\
            méthode de Forward Checking.
        """

        self.psc.consistance_noeuds()
        self.psc.consistance_arcs()
        self.psc.variable_ordering()
        
        self.psc.forward_checking(0, True)
        self.sol = self.psc.solutions

        return self.sol
    
    def affice_solutions(self):
        """ Méthode pour afficher les solutions découvertes par l'algorithme."""
    
        print('Recherche terminée en {} itérations'.format(self.psc.iterations))
    
        if len(self.psc.solutions) == 0:
            print('Aucune solution trouvée')
            return
    
        for sol in self.psc.solutions:
            print('Solution')
            print('========')
            for etat in self.etats:
                print('État {}: '.format(etat.no_etat))
                print('  Propositions initiales:')
                for nom, var in sorted(etat.vars_initiales.items()):
                    if sol[var.nom]:
                        print('    ' + nom)

                print('  Opérateurs:')
                for nom, var in sorted(etat.vars_operateurs.items()):
                    if sol[var.nom]:
                        print('    ' + nom)

                print('  Propositions finales:')
                for nom, var in sorted(etat.vars_finales.items()):
                    if sol[var.nom]:
                        print('    ' + nom)
                print()