from moteur_psc.psc import PSC

class PSCHeuristique(PSC):

    def __init__(self, variables, contraintes):
        """ 
            :param list variables: les variables du problème.
            :param list contraintes: les contraintes du problème.
        """

        PSC.__init__(self, variables, contraintes)

        self.reinitialise()
        
    def reinitialise(self):
        """ Réinitialise les attributs du problème.

            Après l'appel à cette méthode, les labels des variables sont à\
            nouveau égaux aux domaines complets, et les variables ont une\
            valeurs indéfinie (égale à ``None``).
        """

        self.initialise_labels()
        self.solutions = []
        self.iterations = 0

    def initialise_labels(self):
        """ Initialise les labels pour les rendre identiques aux domaines. """

        for var in self.variables:
            var.label = var.domaine[:]

    def consistance_noeuds(self):
        """ Applique la consistance des noeuds sur les contraintes unaires du\
            problème.

            L'algorithme consiste à enlever des domaines de définition toutes\
            les valeurs qui violent les contraintes unaires.
        """

        # Nous appelons d'abord la méthode de la classe-mère PSC pour réduire
        # les domaines.
        PSC.consistance_noeuds(self)

        # Puis, nous nous assurons que les labels sont identiques aux domaines.
        self.initialise_labels()

    def variable_ordering(self):
        """ Trie les variables par ordre croissant de taille de domaine."""

        self.variables.sort(key=lambda x: len(x.domaine))

    def dynamic_variable_ordering(self, k):
        """ Place en position ``k`` la variable non instanciée dotée du label le\
            plus restreint.

            :param k: profondeur actuelle de la recherche.
        """

        index = k
        taille_plus_petite = len(self.variables[index].label)
        for i in range(k + 1, len(self.variables)):
            if len(self.variables[i].label) < taille_plus_petite:
                index = i
                taille_plus_petite = len(self.variables[i].label)
        if k!=index:
            self.variables[k], self.variables[index] = self.variables[index], self.variables[k]


    def propagation_consistante(self, k):
        """ Propage la valeur de la variable actuelle sur les variables\
            suivantes.

            Pour chaque contrainte portant sur la variable courante et sur une\
            ou plusieurs des variables non encore instanciée, appelle la methode\
            ``propage`` de la contrainte pour réduire le label de la deuxième\
            variable.

            :param k: profondeur actuelle de la recherche.
            :return: ``True`` si la valeur de la dernière variable instanciée\
            n'empêche pas l'instanciation des variables suivantes.
        """

        for contrainte in self.contraintes:
            if self.variables[k] in contrainte.variables:
                for i in range(k+1, len(self.variables)):
                    if self.variables[i] in contrainte.variables:
                        if contrainte.propage(self.variables[k]):
                            break
                        else:
                            return False
        return True

    def forward_checking(self, k=0, une_seule_solution=False):
        """ Algorithme du Forward Checking.

            Le Forward Checking essaie de limiter les retours en arrière en\
            restreignant les domaines des variables non instanciées\
            (par application de la consistance des arcs à chaque itération).

            :param une_seule_solution: retourne après avoir trouvé la première\
            solution.
            :param k: la profondeur actuelle de la recherhe.
        """

        if len(self.solutions) == 1 and une_seule_solution:
            return
        self.iterations += 1
        if k >= len(self.variables):
            sol = {}
            for var in self.variables:
                sol[var.nom] = var.val
            self.solutions.append(sol)
        else:
            self.dynamic_variable_ordering(k)
            variable = self.variables[k]

            sauvergarde_labels = {var:  var.label[:] for var in self.variables}

            for val in sauvergarde_labels[variable]:
                variable.val = val
                variable.label = [val]
                if self.propagation_consistante(k):
                    self.forward_checking(k + 1, une_seule_solution)
                for var in self.variables:
                    var.label = list(sauvergarde_labels[var])
            variable.val = None
