
class Contrainte:
    """ Modélisation d'une contrainte abstraite."""
    
    def __init__(self, variables):
        """
            :param list variables: les variables concernées par la contrainte.
        """

        self.variables = tuple(variables)

    def dimension(self):
        """
            :return: le nombre de variables concernées par la contrainte.
        """

        return len(self.variables)

    def est_valide(self):
        """ Teste si la contrainte est respectée par les valeurs actuelles.

            :return: ``True`` si la contrainte est valide.
        """

        return False

    def __repr__(self):
        return 'Contrainte: {}'.format(self.variables)

    def __eq__(self, that):
        return self.variables == that.variables

    def __hash__(self):
        return sum([v.__hash__ for v in self.variables])

class ContrainteUnaire(Contrainte):
    """ Contrainte imposant une restriction sur la valeur d'une variable.
        
        Exemples: ``x > 0``, ``y = 5``.
    """

    def __init__(self, var, op):
        """ Exemples d'op: ``lambda x: x > 5``, ``lambda x: x > 5 and x < 10``.

            :param var: variable concernée par la contrainte.
            :param op: fonction ou expression lambda permettant de vérifier la\
            contrainte.
        """

        Contrainte.__init__(self, (var,))
        self.op = op
    
    def est_valide(self, val):
        """ Teste si la contrainte est valide quand la variable ``var``\
            prend la valeur ``val``.
            
            La contrainte unaire est respectée si l'opérateur appliqué à\
            l'opérande ``val``  retourne ``True``.
            
            :return: ``True`` si la contrainte est valide.
        """
        return self.op(val)

class ContrainteBinaire(Contrainte):
    """ Contrainte imposant une restriction sur deux variables.

        Exemple: ``x > y``.
    """

    def __init__(self, var1, var2, op):
        """ Exemples d'op: ``lambda x,y: x != y``, ``lambda x,y: x < y``.

            :param var1: première variable concernée par la contrainte.
            :param var2: deuxième variable concernée par la contrainte.
            :param op: fonction ou expression lambda permettant de vérifier la\
            contrainte.
        """

        Contrainte.__init__(self, (var1, var2))
        self.op = op

    def est_valide(self, var, val):
        """ Teste si la contrainte est valide quand la variable ``var``\
            (soit ``var1``, soit ``var2``) prend la valeur ``val``.

            La contrainte unaire est respectée si l'opérateur appliqué aux\
            opérandes ``val, var2.val`` (lorsque ``var`` est ``var1``) ou\
            ``var1.val, val`` (lorsque ``var`` est ``var2``) retourne ``True``.

            :return: ``True`` si la contrainte est valide.
        """
        var1, var2 = self.variables

        if var1 == var:
            return self.op(val, var2.val)
        elif var2 == var:
            return self.op(var1.val, val)
        else:
            # var n'est pas une des variables de la contrainte.
            raise ValueError('Mauvaise variable: ' + var.nom + '. ' + 
                             'On attendrait ' + var1 + ' ou ' + var2)

    def est_possible(self, var):
        """ Teste si le domaine de ``var`` contient au moins une valeur\
            satisfaisant la contrainte.

            :return: ``True`` s'il existe au moins une valeur satisfaisant\
            la contrainte.
        """
        if var not in self.variables:
            # var ne fait pas partie des variables de la contrainte
            var1, var2 = self.variables
            raise ValueError('Mauvaise variable: ' + var.nom + '. ' + 
                             'On attendrait ' + var1 + ' ou ' + var2)

        for val in var.domaine:
            if self.est_valide(var, val):
                # Il suffit d'une valeur valide.
                return True

        # Aucune valeur val du domaine n'a retourné True pour 
        # est_valide(var, val).
        return False

    def reviser(self):
        """ Algorithme de révision des domaines.

            Pour chaque variable, vérifie chaque valeur du domaine. Supprime\
            les valeurs qui empêchent la contrainte d'être satisfaite dans le\
            domaine.

            :return: ``True`` si un des domaines a été modifié.
        """
        domaines_modifies = False

        # reversed() retourne l'inverse d'une liste ou d'un tuple.
        # Les paires sont donc (var1, var2) et (var2, var1).
        # Les tuples ne sont pas identiques mais ils contiennent des références 
        # sur les mêmes objets Variable (modifier var1 dans le premier tuple\
        # modifie var1 dans le second).
        for var1, var2 in (self.variables, reversed(self.variables)):
            ancienne_valeur = var1.val
            for val in var1.domaine[:]:
                var1.val = val

                if not self.est_possible(var2):
                    var1.domaine.remove(val)
                    domaines_modifies = True
            var1.val = ancienne_valeur

        return domaines_modifies