/**************************************************************/
/*                                                            */
/*                        Classe WORK                         */
/*                                                            */
/**************************************************************/


import java.util.*;


/**
* Classe Work :
* Espace de travail de la recherche. Contient les methodes principales
* de recherche (DoSimpleSearch, DoStarSearch).
* Utilisee pour l'exercice de recherche de chemins dans un graphe.
*
* @version 	1.0, 1 Avr 1997
* @author Bruno Sakoto
**/

public class Work
{
	// Definition des chaines de caracteres pour les differentes methodes.
	public static final String LARGEUR_DABORD = new String( "Breadth first" );
	public static final String PROFONDEUR_DABORD = new String( "Depth first" );
	public static final String A_STAR = new String( "A star" );


	protected Graph graph;   // Graphe sur lequel on va travailler.


	
	/**************************************************************/
	/* Methode WORK                                               */
	/**************************************************************/

	public Work()
	{
		graph = new Graph();
	}
	
	
	
	/**************************************************************/
	/* Methode DO_SIMPLE_SEARCH                                   */
	/**************************************************************/
	/**
	* Effectue la recherche en profondeur ou largeur d'abord.
	* @param villeDep la ville de depart.
	* @param villeArr la ville d'arrivee.
	* @return vrai si la ville d'arrivee a pu etre touvee et faux sinon.
	**/

	public boolean DoSimpleSearch( SearchApplet applet )
	{
		// Determiner les paneaux de controle et de resultat
		ControlPanel control = applet.control;
		ResultPanel result = applet.result;
	
		// Efacer les chemins qui apparaitraient dans le graphe.
		graph.InitPaths();
		applet.dessin.showPath = false;
		applet.dessin.repaint();
	
				
		// Efacer les resultats et initialiser la nouvelle chaine de resultat
		result.InitDetails();
		StringBuffer details = new StringBuffer( "" );
	
			
		// Determiner les parametres de la recherche
		Town villeDep = graph.FindTown( control.depChoice.getSelectedItem() );
		Town villeArr = graph.FindTown( control.arrChoice.getSelectedItem() );
		String methode = new String( control.algoChoice.getSelectedItem() );
		boolean stepByStep = control.stepCheckbox.getState();
		


		// Debut de la recherche.

		Vector closedList = new Vector();   // La liste des villes deja visites. 
		Vector openList = new Vector();     // La liste des noeuds a explorer.
		
		Node noeudDep = new Node();
		noeudDep.description = villeDep;
		noeudDep.cost = 0;
		
		openList.addElement( noeudDep );
		
		int steps = 1;
	
		while ( ! openList.isEmpty() )
		{
			Node noeudCour = (Node)openList.firstElement();
			openList.removeElementAt( 0 );
				
			if ( ! closedList.contains( noeudCour.description ) )
			{
				if ( noeudCour.IsSolution( villeArr ) )
				{
					// Ecrire l'etape dans le panneau de resultats
					AppendLastStepToString( details, noeudCour, steps );
					
					// Preciser le chemin dans le graphe
					graph.CreatePath( noeudCour );
					
					if ( ! stepByStep )
					{	
						// Mettre a jour le panneau des resultats
						result.SetDetails( details.toString() );
						
						// Redessiner le graphe pour y faire apparaitre le chemin.
						applet.dessin.showPath = true;
						applet.dessin.repaint();
					}
					else
					{
						// Placer les differentes etapes dans la string tokenizer du panneau
						// de resultat.
						result.strTok = new StringTokenizer( details.toString(), "\n" );
						
						// Placer la 1ere etape dans le panneau de resultat.
						result.detailsTextArea.appendText( result.strTok.nextToken() );
						result.detailsTextArea.appendText( "\n" );
			
						if ( result.strTok.hasMoreElements() )
						{
							// Desactiver le panneau de controle.
							control.okButton.disable();
							control.disable();
						
							// Activer les boutons pour l'execution pas a pas.
							result.endButton.enable();
							result.nextButton.enable();
						}
					
					}
					
					return true;
				}
								
				// Ajouter la ville de noeudCour a closedList.
				closedList.addElement( noeudCour.description );
				
				// Nouveaux noeuds <- successeurs de noeudCour.
				Vector nveauxNoeuds = noeudCour.Successors( villeArr, methode );
				
				// Ajouter les nouveaux noeuds a openList.
				AppendNodes( openList, nveauxNoeuds, methode );
				
				// Ecrire l'etape dans le panneau de resultats.
				AppendNewStepToString( details, noeudCour, openList, steps, methode );
				
				steps ++;
			}
		}
				
		return false;
	
	}




	/**************************************************************/
	/* Methode DO_STAR_SEARCH                                     */
	/**************************************************************/
	/**
	* Effectue la recherche a l'aide de l'algoritme A star.
	* @param villeDep la ville de depart.
	* @param villeArr la ville d'arrivee.
	* @return vrai si la ville d'arrivee a pu etre touvee et faux sinon.
	**/
	
	public boolean DoStarSearch( SearchApplet applet )
	{
		// Determiner les paneaux de controle et de resultat
		ControlPanel control = applet.control;
		ResultPanel result = applet.result;
	
		// Efacer les chemins qui apparaitraient dans le graphe.
		graph.InitPaths();
		applet.dessin.showPath = false;
		applet.dessin.repaint();
	
				
		// Efacer les resultats et initialiser la nouvelle chaine de resultat
		result.InitDetails();
		StringBuffer details = new StringBuffer( "" );
	
			
		// Determiner les parametres de la recherche
		Town villeDep = graph.FindTown( control.depChoice.getSelectedItem() );
		Town villeArr = graph.FindTown( control.arrChoice.getSelectedItem() );
		String methode = A_STAR;
		boolean stepByStep = control.stepCheckbox.getState();
		


		// Debut de la recherche.

		Vector closedList = new Vector();   // La liste des villes deja visites. 
		Vector openList = new Vector();     // La liste des noeuds a explorer.
		
		Node noeudDep = new Node();
		noeudDep.description = villeDep;
		noeudDep.cost = 0;
		openList.addElement( noeudDep );
		
		Node noeudCour = noeudDep;
		
		int steps = 1;
	
		while ( ! openList.isEmpty() && ! noeudCour.IsSolution( villeArr ) )
		{
			openList.removeElement( noeudCour );
			closedList.addElement( noeudCour );
			

			
			// Parcourir les successeurs du noeud courant	
			Vector listSucc = noeudCour.Successors( villeArr, methode );
			Enumeration enumSucc = listSucc.elements();
			while ( enumSucc.hasMoreElements() )
			{
				Node succNode = (Node)enumSucc.nextElement();
				
				// Regarder si succNode apparait dans l'open list (ie s'il a deja ete genere mais
				// pas encore traite
				
				Node oldSuccNode = null;
				
				// Parcourir open list 
				Enumeration enumOpen = openList.elements();
				while ( enumOpen.hasMoreElements() && oldSuccNode == null )
				{
					Node n = (Node)enumOpen.nextElement();				
					
					if ( succNode.description.equals( n.description ) )
						oldSuccNode = n;	
				}
				
				// Si succNode n'etait pas dans l'open list regarder s'il n'est pas dans la closed
				// list.
				
				if ( oldSuccNode == null )
				{
					// Parcourir la closed list
					Enumeration enumClosed = closedList.elements();
					while ( enumClosed.hasMoreElements() && oldSuccNode == null )
					{
						Node n = (Node)enumClosed.nextElement();
						if ( succNode.description.equals( n.description ) )
							oldSuccNode = n;
					}
						
				}
				
				
				if ( oldSuccNode == null )
				{
					// succNode n'appartient ni a open list, ni a closed list, l'inserer dans open list

					Vector v = new Vector();
					v.addElement( succNode );
					AppendNodes( openList, v, methode );
				}
				else	
				{
					if ( oldSuccNode.cost > succNode.cost )
					{
						// succNode a deja ete traite, mais on a trouve un meilleur chemin
						
						oldSuccNode.ancestor = noeudCour;
						oldSuccNode.cost = succNode.cost;
						oldSuccNode.finalCost = succNode.finalCost;
						
						if ( openList.contains( oldSuccNode ) )
							openList.removeElement( oldSuccNode );
						else
							closedList.removeElement( oldSuccNode );
						
						// Inserer oldSuccNode dans openList	
						Vector v = new Vector();
						v.addElement( oldSuccNode );
						AppendNodes( openList, v, methode );

					}
				}
			}
								
			// Ecrire l'etape dans le panneau de resultats.
			AppendNewStepToString( details, noeudCour, openList, steps, methode );
				
			steps ++;
			
			if ( ! openList.isEmpty() )
				noeudCour = (Node)openList.firstElement();
				
		}
	
	
		if ( noeudCour.IsSolution( villeArr ) )
		{
			// Ecrire l'etape dans le panneau de resultats
			AppendLastStepToString( details, noeudCour, steps );
					
			// Preciser le chemin dans le graphe
			graph.CreatePath( noeudCour );
					
			if ( ! stepByStep )
			{	
				// Mettre a jour le panneau des resultats
				result.SetDetails( details.toString() );
						
				// Redessiner le graphe pour y faire apparaitre le chemin.
				applet.dessin.showPath = true;
				applet.dessin.repaint();
			}
			else
			{
				// Placer les differentes etapes dans la string tokenizer du panneau
				// de resultat.
				result.strTok = new StringTokenizer( details.toString(), "\n" );
				
				// Placer la 1ere etape dans le panneau de resultat.
				result.detailsTextArea.appendText( result.strTok.nextToken() );
				result.detailsTextArea.appendText( "\n" );
		
				if ( result.strTok.hasMoreElements() )
				{
					// Desactiver le panneau de controle.
					control.okButton.disable();
					control.disable();
				
					// Activer les boutons pour l'execution pas a pas.
					result.endButton.enable();
					result.nextButton.enable();
				}
					
			}
					
			return true;
		}
	
		return false;
	
	}



	

	/**************************************************************/
	/* Methode APPEND_NEW_STEP_TO_STRING                          */
	/**************************************************************/
	/**
	* Ajoute une nouvelle etape de recherche dans une chaine de caracteres.
	* @param str la chaine dans laquelle on ajoute l'etape.
	* @param noeud le noeud de l'etape courante.
	* @param list la liste des noeud a etudier (open list).
	* @param nb le numero de l'etape.
	* @param methode la methode utilisee.
	**/

	public void AppendNewStepToString( StringBuffer str, Node noeud, Vector list, int nb,
	                                   String methode )
	{
		if ( ! str.toString().equals( "" ) )
			str.append( "\n" );
			
		if ( nb < 10 )
			str.append( " " + nb + " - " );
		else
			str.append( nb + " - " );
			
 		str.append( "Node : " + noeud.description.name );
 	
 		if ( ! list.isEmpty() )
 		{
 			str.append( " - List : " );
 			Enumeration enum = list.elements();
 			while ( enum.hasMoreElements() )
 			{
 				Node noeudCour = (Node)enum.nextElement();
 				str.append( noeudCour.description.name ); 
 				
 				if ( methode.equals( A_STAR ) )
 				{
 					str.append( "(" + noeudCour.cost + "+" + noeudCour.description.heuristic +
 					            "=" + noeudCour.finalCost + ") "  );
 				}
 				
 				str.append( " " );
 				
 			}
 		}
		
	}



	/**************************************************************/
	/* Methode APPEND_LAST_STEP_TO_STRING                         */
	/**************************************************************/
	/**
	* Ajoute la derniere etape de recherche dans une chaine de caracteres.
	* @param str la chaine dans laquelle on ajoute l'etape.
	* @param noeud le noeud solution.
	* @param nb le numero de l'etape.
	**/

	public void AppendLastStepToString( StringBuffer str, Node noeud, int nb )
	{
		if ( ! str.toString().equals( "" ) )
			str.append( "\n" );
			
		if ( nb < 10 )
			str.append( " " + nb + " - " );
		else
			str.append( nb + " - " );
			
 		str.append( "Node : " + noeud.description.name + " - Total cost : " + noeud.cost );
 	
 	}
 	


	/**************************************************************/
	/* Methode APPEND_NODES                                       */
	/**************************************************************/
	/**
	* Ajoute un vecteur de noeuds a un autre vecteur. L'emplacement
	* d'insertion depend de la method utilisee.
	* @param list le vecteur auquel on ajoute les nouveaus noeuds.
	* @param nveauxNoeuds le vecteur des nouveaux noeuds a ajouter.
	* @param methode la methode de recherche utilisee.
	**/

	private void AppendNodes( Vector list, Vector nveauxNoeuds, String methode )
	{
		Enumeration enumNode = nveauxNoeuds.elements();
		int lastPos = list.size();

		if ( methode.equals( PROFONDEUR_DABORD ) )
		{
			// Inserer les nouveaux noeuds en tete de list.
			while ( enumNode.hasMoreElements() )
				list.insertElementAt( (Node)enumNode.nextElement(), 0 );
		}
				
		if ( methode.equals( LARGEUR_DABORD ) )
		{
			// Inserer les nouveaux noeuds en queue de list.
			while ( enumNode.hasMoreElements() )
				list.insertElementAt( (Node)enumNode.nextElement(), lastPos );
		}
		
		if ( methode.equals( A_STAR ) )
		{
			// Inserer les nouveaux noeuds dans list puis trier list.
			while ( enumNode.hasMoreElements() )
				list.insertElementAt( (Node)enumNode.nextElement(), list.size() );
			SortNodes( list );		
		}
			
	}



	/**************************************************************/
	/* Methode SORT_NODES                                         */
	/**************************************************************/
	/**
	* Trie des noeuds selon leur cout croissant. Utilise la methode
	* du tri a bulles.
	* @param vNode le vecteur de noeuds a trier.
	**/
	
	private void SortNodes( Vector vNode )
	{
		boolean modif;
		do
		{
			modif = false;
			
			for ( int i=0; i<vNode.size()-1; i++ )
			{
				Node n1 = (Node)vNode.elementAt( i );
				Node n2 = (Node)vNode.elementAt( i+1 );
				if ( n1.finalCost > n2.finalCost )
				{
					vNode.setElementAt( n2, i );
					vNode.setElementAt( n1, i+1 );
					modif = true; 
				}
			}
		}
		while ( modif );
	
	}
	
	
}