from moteur_psc_heuristique.variable_avec_label import VariableAvecLabel
from moteur_psc.contrainte import ContrainteUnaire
from moteur_psc_heuristique.contrainte_avec_propagation import ContrainteAvecPropagation
from moteur_psc_heuristique.psc_heuristique import PSCHeuristique

class Sudoku:
    """ Représentation et résolution d'une grille de Sudoku."""

    def __init__(self, grille, taille=9, sous_taille=3):
        """
            :param taille: taille de la grille de Sudoku.
            :param sous_taille: taille des sous-grilles du problème.
        """

        if taille % sous_taille != 0:
            raise ValueError('Taille et sous-taille de grille incompatibles.')
        self.taille = taille
        self.sous_taille = sous_taille

        # Génère une variable par case.
        self.variables = [VariableAvecLabel('{}{}'.format(i, j), 
                                            list(range(1, self.taille + 1)))
            for i in range(self.taille) for j in range(self.taille)]

        # Initialise les cases dont les valeurs sont connues.
        for i in range(self.taille):
            for j in range(self.taille):
                # Si la case est instanciée avec une valeur valide,
                # assigne la variable et restreint son domaine à la valeur initiale.
                if isinstance(grille[i][j], int):
                    if not (1 <= grille[i][j] <= self.taille):
                        raise ValueError('Valeur invalide dans la grille de départ.')
                    var = self.variables[i * self.taille + j]
                    var.val = grille[i][j]
                    var.domaine = [grille[i][j]]
                    var.label = [grille[i][j]]

        self.contraintes = []
        self.genere_contraintes()

    def __genere_contraintes_sous_grille(self, x, y):
        """ Génère les contraintes de la sous-grille dont le coin est situé\
            en (x,y).
        """

        # Parcours de la sous-grille.
        for i in range(x, x + self.sous_taille):
            for j in range(y, y + self.sous_taille):
                # Pour chaque case qui n'est ni dans la même ligne i ni dans la même colonne j,
                # on ajoute une contrainte. (les autres cases sont couvertes par les contraintes
                # de lignes et de colonnes.)
                for k in range(x, x + self.sous_taille):
                    for l in range(y, y + self.sous_taille):
                        if i != k and j != l:
                            self.contraintes.append(
                                ContrainteAvecPropagation(self.variables[i * self.taille + j],
                                                          self.variables[k * self.taille + l],
                                                          lambda x,y: x != y)
                            )

    def genere_contraintes(self):
        """ Génère toutes les contraintes d'une grille de Sudoku. """

        self.contraintes = []

        for i in range(0, self.taille):
            for j in range(0, self.taille):
                # Contraintes sur les case d'une ligne.
                for k in range(j + 1, self.taille):
                    self.contraintes.append(
                        ContrainteAvecPropagation(self.variables[i * self.taille + j],
                                                  self.variables[i * self.taille + k],
                                                lambda x,y: x != y)
                    )
                # Contraintes sur les cases d'une colonne.
                for k in range(j + 1, self.taille):
                    self.contraintes.append(
                        ContrainteAvecPropagation(self.variables[j * self.taille + i],
                                                  self.variables[k * self.taille + i],
                                                  lambda x,y: x != y)
                    )

        # Contrainte sur les cases d'une sous-grille.
        # Le troisième argument de range permet de régler l'incrément.
        # Ex.: range(0, 5, 2) génère la séquence 0, 3.
        #      range(10, 5, -1) génère la séquence 10, 9, 8, 7, 6.
        # '//' est l'opérateur de division entière.
        for i in range(0, self.taille, self.taille // self.sous_taille):
            for j in range(0, self.taille, self.taille // self.sous_taille):
                self.__genere_contraintes_sous_grille(i, j)

    def resoudre(self, methode):
        """ Lance l'algorithme de résolution choisi sur la grille.

            :param methode: méthode de résolution: ``'forward_checking'`` ou\
            ``'backtracking'``.
        """

        psc = PSCHeuristique(self.variables, self.contraintes)
        psc.consistance_noeuds()
        psc.consistance_arcs()

        if methode == 'forward_checking':
            psc.forward_checking(une_seule_solution=True)            
        elif methode == 'backtracking':
            psc.variable_ordering()
            psc.backtracking(une_seule_solution=True)
        else:
            raise ValueError('Méthode inconnue: ' + str(methode))
        print('Méthode: ' + methode)
        print('Recherche terminée en {} itérations'.format(psc.iterations))
        for i in range(self.taille):
            for j in range(self.taille):
                nom = '{}{}'.format(i, j)
                self.variables[i * self.taille + j].val = psc.solutions[0][nom]

    def __repr__(self):
        """ Convertit les variables du problème en grille à afficher."""

        def val(e):
            """ Méthode locale pour la mise en forme des valeurs absentes."""

            if e is None:
                return '-'
            else:
                return e

        ret = ''
        for i in range(self.taille):
            for j in range(self.taille):
                ret += '{} '.format(val(self.variables[i * self.taille + j].val))
            ret += '\n'

        return ret