/**************************************************************/
/*                                                            */
/*                       Classe WORK                          */
/*                                                            */
/**************************************************************/


package utils;

import JCL.*;
import graphics.*;
import java.util.*;


/**
* Classe Work :
* Espace de travail pour le probleme d'affectation des ressources.
*
* @version 1.0, 1 May 1997
* @author Bruno Sakoto
**/

public class Work
{
	// Definition des constantes
	protected static final int DELAY_MAX = 100;
	protected static final int DUREE_MAX = 125;
	protected static final int DUREE_MIN = 45;
	protected static final int T_MAX = 500;
	
	public static final int NB_RESOURCES = 6;  
	public static final String[] T_RES_NAME = { "a", "b", "c", "d", "e", "f" }; 
	
	public static final String[] T_ALGO = { "BT", "FC", "FC with DVO", "MAC", "MAC with DVO",
	                                        "BJ", "CBJ" };
  
  
	protected MyRandom r;           // Le generateur de nombres aleatoires.
  	protected Vector vTask;         // L'ensemble des taches a effectuer.     

  	protected TasksCanvas dessin;   // Le graphe dans dans lequel sont representees les taches.
  	protected ControlPanel control; // Le panneau de controle.
  	protected InfoPanel info;       // Le panneau d'information
  	protected Network net;          // le reseau pour le CSP
	protected Solver solver;        // Le solver.
	
	public Monitor monitor;
	


	/**************************************************************/
	/* Methode WORK                                               */
	/**************************************************************/

	public Work( TasksCanvas d, ControlPanel ctrl, InfoPanel i )
	{
		r = new MyRandom();
		vTask = new Vector();
		
		dessin = d;
		control = ctrl;
		info = i;
				
		monitor = new Monitor();
	}




	/**************************************************************/
	/* Methode GET_TASKS                                          */
	/**************************************************************/
	/*
	* Permet d'obtenir l'ensemble des taches de l'espace de travail.
	*/

	public Vector GetTasks()
	{
		return this.vTask;
	}
	
	
	
	
	/**************************************************************/
	/* Methode GET_TASK                                           */
	/**************************************************************/
	/*
	* Permet d'obtenir une des taches de l'espace de travail.
	*/
	
	public Task GetTask( int i ) 
	{
		return (Task)this.vTask.elementAt( i );
	}




	/**************************************************************/
	/* Methode GET_DESSIN                                         */
	/**************************************************************/
	/*
	* Permet d'obtenir le dessin des taches de l'espace de travail.
	*/

	public TasksCanvas GetDessin()
	{
		return this.dessin;
	}





	/**************************************************************/
	/* Methode GET_CONTROL                                        */
	/**************************************************************/
	/*
	* Permet d'obtenir le panneau de controle de l'espace de travail.
	*/

	public ControlPanel GetControl()
	{
		return this.control;
	}




	/**************************************************************/
	/* Methode GET_INFO                                           */
	/**************************************************************/
	/*
	* Permet d'obtenir le panneau d'information de l'espace de travail.
	*/

	public InfoPanel GetInfo()
	{
		return this.info;
	}


	/**************************************************************/
	/* Methode GET_SOLVER                                         */
	/**************************************************************/
	/*
	* Permet d'obtenir le solver de l'espace de travail.
	*/

	public Solver GetSolver()
	{
		return this.solver;
	}




	/**************************************************************/
	/* Methode PRINT_TASKS                                        */
	/**************************************************************/
	/*
	* Affiche les taches.
	*/

	public void PrintTasks()
	{
		for ( int i=0; i<vTask.size(); i++ )
		{
			System.out.print( i + "\t" );
			((Task)vTask.elementAt( i )).Print();
		}
	}



	/**************************************************************/
	/* Methode INDEX_OF_RESOURCE                                  */
	/**************************************************************/
	/*
	* Calcule l'index d'une ressource dans le tableau des noms des 
	* ressources.
	* @param resName le nom de la ressource dont on veut l'index.
	* @return l'index recherche ou -1 si aucun.
	*/

	public static int IndexOfResource( String resName )
	{
		for ( int i=0; i<NB_RESOURCES; i++ )
		{
			if ( resName.equals( T_RES_NAME[i] ) )
				return i;
		}
		return -1;
	}





	/**************************************************************/
	/* Methode GENERATE_RESOURCES                                 */
	/**************************************************************/
	/*
	* Genere un ensemble de ressources aleatoires.
	*/
	
	public int[] GenerateResources() throws MyRandomException
	{
		int tRes[] = new int[NB_RESOURCES];

		int bSup = (int)Math.pow( 2, NB_RESOURCES );

		int mask = r.nextIntBetween( 1, bSup );
		
		// Convertir mask en binaire pour avoir les resources qui ont ete choisies.
		for ( int i=NB_RESOURCES-1; i>=0; i-- )
		{
			tRes[i] = mask % 2;
			mask = mask / 2;
		}

		return tRes;

	}




	/**************************************************************/
	/* Methode GENERATE_TASKS                                     */
	/**************************************************************/
	/*
	* Genere un ensemble de taches aleatoires dans l'espace de travail.
	*/

	public void GenerateTasks() throws MyRandomException
	{	
		// Supprimer tous les taches
		vTask.removeAllElements();
		
		int ligne = 0;
		int t0, t1;
      
		while ( ligne < 5 )
		{
			t1 = 0;

			do
			{
				int delay = r.nextIntBetween( 1, DELAY_MAX );
				int d = r.nextIntBetween( DUREE_MIN, DUREE_MAX );

				t0 = t1 + delay;
				t1 = t0 + d;
	      
				if ( t1 < T_MAX )
				{
					Task tache = new Task();
					tache.line = ligne;
					tache.beginTime = t0;
					tache.endTime = t1;
					tache.tResource = GenerateResources(); 
		  
					vTask.addElement( tache );
		  
				}
				else
				{
					ligne ++;
				}		

			}
			while ( t1 < T_MAX );

		}
		
		
		// Initialisation du reseau pour le CSP
		
		TaskNetworkBuilder builder;
		LiteralNetwork literalNet;
			
		System.out.println( "Building the CSP..." );
		builder = new TaskNetworkBuilder( "Resource Allocation","Marc Torrens & Bruno Sakoto", vTask );
		literalNet = builder.Build();
		System.out.println( "Done !" );

		System.out.println( "Building the network..." );
		net = literalNet.BuildNetwork ();
		System.out.println( "Done !" );

	}
	



	/**************************************************************/
	/* Methode INIT_TASKS                                         */
	/**************************************************************/
	/*
	* Initialise les taches. 
	* Les domaines restreints sont restaures et les resources allouees
	* sont supprimees.
	*/

	public void InitTasks()
	{	
		Enumeration enum = vTask.elements();
		while ( enum.hasMoreElements() )
		{
			Task tache = (Task)enum.nextElement();
			
			for ( int i=0; i<Work.NB_RESOURCES; i++ )
			{
				if ( tache.tResource[i] == -1 )
					tache.tResource[i] = 1;
			}
		
			tache.resAlocated = -1;	
		
		}

	}



	/**************************************************************/
	/* Methode SOLVE                                              */
	/**************************************************************/
	/*
	* Initialise le reseau et resoud le probleme.
	*/	
	
	public void Solve()
	{

		// Initialisation des taches
		InitTasks();
		
		// Initialisation de la zone info
		info.InitDetails();

		dessin.repaint();
		

		// Set the algorithm
		String strAlgo = control.GetAlgoChoice().getSelectedItem();
				
		if ( strAlgo.equals( "BT" ) )
			solver = (Solver)new BTSolver( net );
		
		else if ( strAlgo.equals( "FC" ) )
			solver = (Solver)new FCSolver( net );
			
		else if ( strAlgo.equals( "MAC" ) )
			solver = (Solver)new FC_ARCSolver( net );
			
		else if ( strAlgo.equals( "FC with DVO" ) )
			solver = (Solver)new FC_DVOSolver( net );
			
		else if ( strAlgo.equals( "MAC with DVO" ) )
			solver = (Solver)new MAC_DVOSolver( net );

		else if ( strAlgo.equals( "BJ" ) )
			solver = (Solver)new BJSolver( net );
		
		else if ( strAlgo.equals( "CBJ" ) )
			solver = (Solver)new CBJSolver( net );
			
		else
		{
			System.out.println( "Erreur lors du choix de l'algo." );
			System.exit( 1 );
		}
		
			
		info.AppendDetails( "Solving the CSP with the \"" + strAlgo + "\" algorithm ..." );
		info.AppendDetails( "\n" );

		// Set the solution manager
		solver.SetSolutionManager (new MySolutionManager ( this ));      

		// Set the step by step manager
		if ( control.GetStepCheckbox().getState() )
			solver.SetStepByStepManager (new MyStepByStepManager ( this ));

		// Set the number of solutions to find to 'all'
		solver.SetNumberOfSolutionsToFind (1);

		solver.StartSolving ();
			
	}
	
	
	
}
