package JCL;

/**
 * Representation of an abstract problem solver for the Java Constraint
 * Library.
 *
 * @author Erik Bruchez
 */

public abstract class Solver implements Runnable {

	Thread thread;	//	the associated thread of execution

	protected Network net;		//	the network representing the CSP
	protected int indexes[];	//	instanciated variables indexes
	protected int size = 0;		//	shortcut to network size
	protected int MAX_DOMAIN = 0;	//	shortcut to network max domain size

	int find = 1;			//	number of solutions to find
	boolean found = false;		//	at least one solution has been found
	int count = 0;			//	the number of solutions found

	SolutionManagerInterface solution_manager;	//	the solution manager
	StepByStepManager step_manager;			//	the step-by-step manager
	SolutionAttributes attributes;			//	useful solution attributes
	boolean suspended = false;			//	is the solving suspended ?

	/*
	 *	Constructors.
	 */

	public Solver () {
	}

	public Solver (Network net) {
		SetNetwork (net);
	}
	
	/**
	 *	Set the associated network.
	 */

	public void SetNetwork (Network net) {
		int dom = 0;	
	
		this.net = net;
		size = net.GetSize ();
		indexes = new int[size];
		for (int i = 0; i < size; i ++ ) {
			dom = net.GetVariable (i).GetDomain ().GetSize ();
			if (dom > MAX_DOMAIN)
				MAX_DOMAIN = dom;
		}
	}

	/**
	 *	Set the solution manager.
	 */

	public void SetSolutionManager (SolutionManagerInterface manager) {
		solution_manager = manager;
	}

	/**
	 *	Set the step-by-step manager.
	 */

	public void SetStepByStepManager (StepByStepManager manager) {
		step_manager = manager;
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public abstract String GetName ();

	/**
	 *	Set the number of solutions to find. If the value is -1, find
	 *	all the solutions.
	 */

	public void SetNumberOfSolutionsToFind (int n) {
		find = (n >= 0) ? n : -1; 
	}

	/**
	 *	Get the number of solution to find.If the value is -1, find
	 *	all the solutions.
	 */

	public int GetNumberOfSolutionsToFind () {
		return find;
	}

	/**
	 *	Return true if there are more solutions to find.
	 */

	protected boolean FindMoreSolutions () {
		return (find == -1) || (count < find);
	}

	/**
	 *	Runnable implementation.
	 */

	public void run () {
		try {
			NotifyStart ();
			if (find != 0)
				Solve ();
			NotifyEnd ();
		} catch (ThreadDeath e) {
			NotifyEnd ();
			throw e;
		}
	}

	/**
	 *	Solving method.
	 */

	public abstract void Solve ();

	/**
	 *	Start solving the network using a new thread.
	 */

	public synchronized void StartSolving () {
		if (thread == null) {
			thread = new Thread (this, GetName ());
			int prio = Thread.currentThread ().getPriority ();
			thread.setPriority (Math.max (Thread.MIN_PRIORITY, prio - 1));
			thread.start ();
		}
	}

	/**
	 *	Suspend the solving process.
	 */

	public synchronized void SuspendSolving () {
		if (thread != null) {
			thread.suspend ();
			suspended = true;
		}
	}

	/**
	 *	Resume the solving process.
	 */

	public synchronized void ResumeSolving () {
		if (thread != null) {
			thread.resume ();
			suspended = false;
		}
	}

	/**
	 *	Stop the solving process.
	 */

	public synchronized void StopSolving () {
		if (thread != null) {

			//	Necessary / useless / wrong ?

			if (suspended) {
				thread.resume ();
				suspended = false;
			}

			thread.stop ();
			thread = null;
		}
	}

	/**
	 *	Notify the solution manager that a solving is beginning.
	 */

	private void NotifyStart () {
		attributes = new SolutionAttributes (net.GetName (),
			net.GetAuthor (), GetName ());

		if (solution_manager != null)
			solution_manager.NotifyStart (net, this, attributes);
		
		if (step_manager != null)
			step_manager.NotifyStart (net);
	}

	/**
	 *	Notify the solution manager that a new solution has been found.
	 */
	
	protected void NotifySolution () {
		found = true;
		count++;

		if (solution_manager != null) {
		  Solution s = new Solution (net.GetSize(),indexes, attributes);
		  solution_manager.NotifySolution (s);
		}
	}
	
	/**
	 *	Notify the solution manager that the solving is ending.
	 */

	private void NotifyEnd () {
		if (solution_manager != null)
			solution_manager.NotifyEnd ();
		
	}

	/**
	 *	Notify the step-by-step manager that a new level is entered.
	 */

	protected void NotifyEnterLevel (int level) {
		if (solution_manager != null)
			solution_manager.NotifyEnterLevel();
		if (step_manager != null)
			step_manager.NotifyEnterLevel (level);
	}





	/**
	 *	Notify the step-by-step manager that the current level is left.
	 */

	protected void NotifyLeaveLevel (int level) {
		if (solution_manager != null)
			solution_manager.NotifyLeaveLevel();
		if (step_manager != null)
			step_manager.NotifyLeaveLevel (level);
	}


	protected void NotifyLeaveLevelAux (int level) {
		if (step_manager == null)
			return;
		else
			step_manager.NotifyLeaveLevelAux (level);
	}



	protected void NotifyBackjump () {
		if (solution_manager != null)
			solution_manager.NotifyBackjump();
		if (step_manager != null)
			step_manager.NotifyBackjump();
	}


	/**
	 *	Notify the step-by-step manager that a consistency check is done.
	 */

	protected void NotifyConsistencyCheck () {
		if (solution_manager != null)
			solution_manager.NotifyConsistencyCheck();
		if (step_manager != null)
			step_manager.NotifyConsistencyCheck();
	}

	/**
	 *	Notify the step-by-step manager that a variable is instanciated.
	 */

	protected void NotifyInstanciation (int variable, int value){
		if (solution_manager != null)
			solution_manager.NotifyInstanciation();
		if (step_manager != null)
			step_manager.NotifyInstanciation (variable, value);
	
	}

	/**
	 *	Notify the step-by-step manager that variable v1 causes v2 domain to
	 *	be restricted. domain contains only -1's, except for values that
	 *	have been removed from the domain, where the number of the responsible
	 *	variable is kept.
	 */

	protected void NotifyDomainRestriction (int v1, int v2, int domain[],
		int old_size, int shrink) {

		if (step_manager == null)
			return;
		else
			step_manager.NotifyDomainRestriction (v1, v2, domain, old_size, shrink);
	}

	/**
	 *	Notify the step-by-step manager that the v2 domain restriction caused by
	 *	variable v1 is restored with restore_count values.
	 */

	protected void NotifyDomainRestoration (int v1, int v2, int domain[], int restore_count) {
		if (step_manager == null)
			return;
		else
			step_manager.NotifyDomainRestoration (v1, v2, domain, restore_count);
	}
}
