package JCL;

/**
 * Implementation of a simple Path Consistency achieving solver 
 * for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version 26-11-96
 */

public class PCSolver extends SolverConsistency {
	
	boolean found = false;
	int count = 0;

	final static String NAME = "Path Consistency";
	
	int stack [][];
	boolean on_stack [][];
	int top = -1;
	
	boolean U[][];
	final static int MAX_DOMAIN = 30;
	boolean values []; 
	
	/*
	 *	Constructors.
	 */

	public PCSolver () {
	}

	public PCSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		try {
			PathConsistency ();
		} catch (SolutionFoundException e) {
		}
	}

	/*
	 *	Perform path consistency on the network.
	 */

	private void PathConsistency () {

		//	Perform path consistency

		ac = Check ();

		super.SetNetwork (this.net);
	}

	/*
	 *	Function to perform complete arc consistency on entire constraint network.
	 *	The process considers each "half" of the constraint network separate as
	 *	also done by Mackworth. It processes each edge using a stack to store
	 *	edges still to be examined. It first pushes all "edges" to be examined.
	 *	Then, it processes the edges one by one, placing variables back on the
	 *	stack if necessary (i.e. their domain values have changed). Note that
	 *	edges simply correspond to the edges leading into the variable stored on
	 *	the stack (thus only these variables need to be stored instead of all
	 *	edges). The function returns whether the network is consistent or not by
	 *	checking if any variables have their domains empty. Note that if a variable
	 *	x has any of its domain values eliminated, this is noted in the constraint
	 *	C[x][x], by eliminating this value for the original "I" matrix. [CSPLib]
	 */

	private boolean Check () {


		int i, j, k, a;

		ClearStack ();

		


		DetUs ();


		

		/*

		 *	First apply Floyd-Warshall verison of path consistency,

		 *	pushing any edges that have changed. [CSPLib]

		 */

		
		int pushes = 0;
		for ( k = 0; ((k < size) && (!DomainsEmpty ())); k ++) 
		for ( i = 0; ((i < size) && (!DomainsEmpty ())); i ++) if ( !U [i][k] )
		for ( j = i; ((j < size) && (!DomainsEmpty ())); j ++) if ( !U [k][j] ) {
			if ( ComposeAndIntersect ( i, k , j )) {
				pushes ++;
				Push (i,j);
				U [i][j] = false;
				U [j][i] = false;
			}
		}
		
		
		/*
		 *	Now process the edges that were pushed. [CSPLib]
		 */
		 
		 while ( !StackEmpty () && (!DomainsEmpty ()) ) {
		 	i = Pop_x ();
		 	j = Pop_y (i);
		 	for (k = 0; k < size; k ++) {
		 		if (!DomainsEmpty ()) {
		 		if ( !U [j][k] && ComposeAndIntersect (i, j, k) ) {
		 		
		 			Push (i, k);
		 			U [i][k] = false;
		 			U [k][i] = false;
		 		}}
		 		if (!DomainsEmpty ()) {
		 		if ( !U [k][i] && ComposeAndIntersect (k, i, j) ) {
		 		
		 			Push (k, j);
		 			U [k][j] = false;
		 			U [j][k] = false;
		 		}}
		 	}
		 }	
		
		
		for (i = 0; i < size; i++) {
			int count = 0;

			Variable v = net.GetVariable (i);
			Domain d = v.GetDomain ();

			int i_domain_size = d.GetSize ();

			for (j = 0; j < i_domain_size; j++)
				if (net.GetConstraint (i, i).GetConstraint (j, j))
					count++;

			if (count == 0) {
				System.out.println ("Warning ! The variable " + i + " has no values !");
				return false;
			}
		}

		return true;
	}
	
	
	/*
 	 *  Stack functions for path consistency where only edges have to be stored.
	 *  Push stores an edge on the stack and marks it as being on the stack.  If it
	 *  is already on the stack, it is not added again.  Pop removes the top edge
	 *  from the stack and unmarks it appropriately.  The stack_empty macro returns
	 *  whether the stack is empty or not.
	 */

	
	/*
	 *	Clear and create the stack.
	 */

	private void ClearStack () {
		if (stack == null)
			stack = new int [(size * (size + 1)) / 2][2];
		if (on_stack == null)
			on_stack = new boolean [size][size];

		top = -1;

		for (int i = 0; i < size; i++)
		        for (int j = 0; j < size; j ++)
			      on_stack [i][j] = false;
	}

	/*
	 *	Push an couple of integers on the stack.
	 */

	private void Push (int x, int y) {
		if (!on_stack[x][y]) {
			top++;
			stack [top] [0] = x;
			stack [top] [1] = y;
			on_stack [x] [y] = true;
			on_stack [y] [x] = true;
		}
	}

	/*
	 *	Pop an integer (a) from the stack.
	 */

	private int Pop_x () {
		int x;
		x = stack [top] [0];
		return x;
	}
	
	private int Pop_y (int x) {
		int y;
		y = stack [top][1];
		on_stack [y][x] = false;
		on_stack [x][y] = false;
		top --;
		return y;	
	}

	/*
	 *	Return true if the stack is empty.
	 */

	private boolean StackEmpty () {
		return top == -1;
	}


	private boolean ComposeAndIntersect (int i, int k, int j) {
	
		boolean result;
		int a, b, c, changes = 0;
		Constraint con;
		values = new boolean [MAX_DOMAIN];
		for (a=0; a < MAX_DOMAIN; a++) 
			values[a] = false;
		
		int domA = net.GetVariable (i).GetDomain ().GetSize (); 
		int domB = net.GetVariable (j).GetDomain ().GetSize ();
		int domC = net.GetVariable (k).GetDomain ().GetSize ();
		
		for (a = 0; a < domA; a ++)
		for (b = 0; b < domB; b ++) {
		
			NotifyValue (0,0,0);
		
			if (net.GetConstraint (i,j).GetConstraint (a, b)) {
			
				result = (net.GetConstraint (i, k).GetConstraint (a, 0)) &
					 (net.GetConstraint (k, k).GetConstraint (0, 0)) &
					 (net.GetConstraint (k, j).GetConstraint (0, b));
				
				for (c = 0; !result && c < domC; c ++)
					result = (net.GetConstraint (i, k).GetConstraint (a, c)) &
						 (net.GetConstraint (k, k).GetConstraint (c, c)) &
						 (net.GetConstraint (k, j).GetConstraint (c, b));
				if (!result) {
				
					changes ++;
					
					if ((i == j) && (a == b)) {
						NotifyRemoveValue (i, a);
						NotifyRemoveValue ();
						values [a] = true;
						con = net.GetConstraint (i, j);
						con.SetExplicitConstraint ();
						con.ClearConstraint (a, b);
						
					} else {
						con = net.GetConstraint (i, j);
						if ((con.IsExplicit ()) && con.GetConstraint (a,b)) { 
							NotifyRemoveConstraint (i, j, a, b);
							NotifyRemoveConstraint ();			
							con.ClearConstraint (a, b);
							con = net.GetConstraint (j, i);
							con.ClearConstraint (b, a);
						}
					}
				}
				 
			}
		
		}
		int del = 0;
		if (net.GetConstraint (i, i).IsExplicit ()) {
			for (b = 0; b < MAX_DOMAIN; b ++) {
				if (values [b]) {
					RemoveValue (i, b - del);
					del ++;				
				}
			}
				
		}
		return (changes != 0);
					
	}

	private void DetUs () {
		int i, j, a, b;
		boolean all_ones;
		int domA , domB;
		
		if (U == null)
			U = new boolean [size][size];
		
		for (i = 0; i < size; i ++) {
			domA = net.GetVariable (i).GetDomain ().GetSize ();
			for (j = 0; j < size; j ++) {
				domB = net.GetVariable (j).GetDomain ().GetSize ();
					
				all_ones = true;
				for (a = 0; a < domA; a ++)
					for (b = 0; ((b < domB) && all_ones) ; b++) {
						if (net.GetConstraint (i,j).IsExplicit ())
							all_ones = all_ones & 
								(net.GetConstraint (i, j).GetConstraint (a, b));
					}
				U[i][j] = all_ones;
			}
		}
	}
	
	/*
	 *	Remove the valor i from the domain of the variable x
	 */

	private void RemoveValue (int x, int i) {
		Domain d;
		Domain d2;
		Variable v;
		Constraint c1,c2, c3;
		int dom1;
		int dom2;
		int l,m;	

		v = net.GetVariable (x);
		d = v.GetDomain ();
		String d2name = new String (d.GetName () + v.GetName ());
		int d2size = d.GetSize () - 1;		//	d2size could be 0
		d2 = new Domain (d2name, d2size);

		int count = 0;
		for (int k = 0; k < d.GetSize (); k ++) {
			if (k != i) {
				d2.SetValueName (count, d.GetValueName (k));
				count++;
			} 
		}
		//	d2 is the new domain for the variable x
		v.SetDomain (d2);

		//	Setup the constraints x, y

		dom1 = net.GetVariable (x).GetDomain ().GetSize ();
		for (int y = 0; y < size; y ++) {
		
			//	Constrain for the test
			c1 = net.GetConstraint (x, y);
			//	Constraint y, x
			c3 = new Constraint (net.GetVariable(y),net.GetVariable(x));
			//	Constraint x, y
			c2 = new Constraint (net.GetVariable(x),net.GetVariable(y));
			for (m = 0; m < dom1; m ++) {
			dom2 = net.GetVariable (y).GetDomain ().GetSize ();
			for (l = 0; l <  dom2; l ++) {
				if ((m == l) && (y == x)) {
					c2.SetConstraint (m, m);
					c3.SetConstraint (m, m);
				} else {
					if (m >= i) {
					
						//	Rotate the constraints
						if (c1.GetConstraint (m + 1, l)) {
							c2.SetConstraint (m, l);
							c3.SetConstraint (l, m);
						} else {
							//c1.ClearConstraint (m, l);
						}
					} else {
						//	Copy the constraints
						if (c1.GetConstraint (m , l)) {
							c2.SetConstraint (m, l);
							c3.SetConstraint (l, m);
						} else {
							//c1.ClearConstraint (m, l);
						}
					}
				}
			}
		}
		net.SetConstraint (c2);
		net.SetConstraint (c3);
		}
	}
	
	private boolean DomainsEmpty () {
		boolean empty = false;
		for (int i = 0; ((i < size) && !empty); i ++) {
			empty = empty || (net.GetVariable (i).GetDomain ().GetSize () == 0);		
		}
		return empty;
	}
	


}
