package JCL;


/**
 * Implementation of a simple Forward Checking with Full Arc Consistency and with
 * Dynamic Variable Ordering solver
 * for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last update 17-4-97
 */

public class MAC_DVOSolver extends Solver {

  /*
   *	Data structures to hold the required information.  See the accompanying
   *	research articles for complete descriptions of the structures.  The
   *	"domains" structure works as follows.  If an entry contains a 0, then the
   *	value is still allowed.  Otherwise, it contains the variable whose
   *	instantiation eliminated it.  Note that this data structure has no
   *	information from preprocessing (which is kept directly in the "C" matrix).
   *	Note that the stack is for the previously mentioned "functions".
   */
  
  int domains[][];
  boolean checking[][];
  
  // The stack
  int stack[];
  boolean on_stack[];
  int top;
  
  // The structures for the DVO     
  int ordering [];
  int connections[];
  int tieBreak [];
  
  int current_domain_size;
  
  final static String NAME = "MAC with Dynamic Variable Ordering";
  final static int MAX_CONSTRAINTS = 100000;
  final static int MAX_DOMAINS = 100000;
  
  /*
   *	Constructors.
   */
  
  public MAC_DVOSolver () {
  }
  
  public MAC_DVOSolver (Network net) {
    SetNetwork (net);
  }

  /**
    *	Return the name of the algorithm.
    */
  
  public String GetName () {
    return new String (NAME);
  }
  
  /**
    *	Try to solve the problem.
    */

  public void Solve () {
    
    //	Initialize variables
    
    domains = new int[size][MAX_DOMAIN];
    checking = new boolean[size][size];
    tieBreak = new int[size];
    stack = new int [size];
    on_stack = new boolean [size];

    ordering = new int [size];
    connections = new int [size];
    
    ClearSetup (size, MAX_DOMAIN);
    
    //	Start recursion
    
    FindNextVar (0);
    int firstVar = ordering [0];

    Domain d = net.GetVariable (firstVar).GetDomain ();
    
    try {
      RecursiveSolve (0, d.GetSize ());
    } catch (SolutionFoundException e) {
    }
  }
	
  /*
   *	Solve the constraint network using the simple FORWARD CHECKING (FC) method.
   *	If a solution is found then notify it and return the value according to
   *	whether the first solution is desired or all solutions. Otherwise, begin
   *	checking each ossible nstantiation of the variable.  For each domain value,
   *	perform the ollowing steps. First, if preprocessing eliminated the value,
   *	disregard the rest of the loop.  Otherwise, instantiate the variable,
   *	check if the network is till consistent, and then call the backtracking
   *	routine recursively. After checking each domain value, restore the domains
   *	which were eliminated by the instantiation. After checking all possible
   *	domain values, return. [CSPLib]
   */
  
  private boolean RecursiveSolve (int current, int k) {
    
    //	Local Variables
    int kk = 0;
    Variable v;
    
    NotifyEnterLevel (current);
    
    if (current >= size) {
      
      //	A solution has been found
      
      NotifySolution ();
      
      if (FindMoreSolutions ()) {
	NotifyLeaveLevel (current);
	return false;
      } else
	throw new SolutionFoundException ();
    }
    
    for (int i = 0; i < k; i++) {
      if ( (net.GetConstraint (ordering[current], ordering[current]).GetConstraint (i, i) == false)
	   || (domains[ordering[current]][i] != -1) )
	continue;
      
      indexes[ordering[current]] = i;
      NotifyInstanciation (ordering[current], i);
      
      if (IsConsistent (current, indexes[ordering[current]])) {
	if (current < (size - 1)) {

	  FindNextVar (current+1);
	  v = net.GetVariable (ordering[current + 1]);
	  kk = v.GetDomain ().GetSize ();
	}
	if (RecursiveSolve (current + 1, kk)) {
	  NotifyLeaveLevel (current);
	  return true;
	}
      }

      Restore (current);
    }
    
    NotifyLeaveLevel (current);
    return false;
  }

  /*
   *	Check if the current instantiation of the variables is consistent by
   *	checking if the current instantiation of a variable will reduce some future
   *	variable's domain to the empty set, in which case the appropriate result is
   *	returned.  For each domain change in a future variable, the appropriate
   *	edges are pushed onto the stack. [CSPLib]
   */
  
  private boolean IsConsistent (int current, int value) {
    
    int lookahead, i, j;
    boolean END = false;
    
    for (i = current + 1; ((i < size) && (!END)); i++) {
      if (CheckForward (ordering[current], ordering [i], value) == 0) {
	// return false;
	END = true;
	break;
      }
    }
    if (!END)
      {
	while ((!StackEmpty ()) && (!END)) {
	  j = Pop ();
	  for (i = current + 1; ((i < size) && (!END)); i ++)
	    if ((ordering[i] != j) && (!END)) {
	      lookahead = Revise (ordering[i], j, ordering [current]);
	      if (lookahead == 0) {
		END = true;
		break;
	      }
	      else if ((lookahead != -1)) {
		Push (ordering[i]);
	      }
	      
	    }
	}
	if (!END) {
	  return true;
	}
      } // END
    
    END : {
      while (!StackEmpty ()) {
	j = Pop ();
      }
      
    }
    return false;
    
  }
  

  /*
   *	Determine which domain values for variable "i" can be eliminated by
   *	considering the edge between "i" and "j".  The number of values left in the
   *	domain of "i" is returned. (-1 indicates no change to the edge).  The
   *	"checking" matrix is appropriately updated. [CSPLib]
   */
  
  private int Revise (int i, int j, int current) {
    int a;
    int domain_count = 0;
    int delete_count = 0;
    
    int k = net.GetVariable (i).GetDomain ().GetSize();
    for (a = 0; a < k; a ++)
      if (domains[i][a] == -1) {
	domain_count ++;
	if (!Exists (net.GetConstraint (i,j), j, a)) {
	  domains[i][a] = current;
	  delete_count ++;
	}
      }
    if (delete_count > 0) {
      checking[current][i] = true;
      NotifyDomainRestriction (j, i, domains[i], domain_count, delete_count);
    }
    return ( (delete_count > 0) ? (domain_count - delete_count) : -1);
  }
  
  /*
   *	This function determines whether given a specific edge, there exists some 
   *	instantiation for "y" given the instantiation for "x". [CSPLib]
   */
  
  private boolean Exists (Constraint con, int y, int a) {
    int b;
    
    int k = net.GetVariable (y).GetDomain().GetSize();
    for (b = 0; b < k; b ++)
      if (domains[y][b] == -1) {
	NotifyConsistencyCheck ();
	if (con.GetConstraint (a , b))
	  return true;
      }
    return false;
  }
  
  
  /*
   *	This function checks the edge between variables "i" and "j" to see
   *	which values in the domain of "j" can be eliminated due to the
   *	current instantiation of "i" as "value". Note that given a variable
   *	"i" and an instantiation "v", domains[i][v] is -1 if the value is
   *	still allowed, otherwise it contains the variable which caused it
   *	to be eliminated. If any value in "j" is deleted, this is noted in
   *	the "checking" matrix. The number of values in the domain of "j"
   *	after all deletions is returned. [CSPLib]
   */

  private int CheckForward (int current, int j, int value) {
    
    int domain_count = 0;
    int delete_count = 0;
    
    Variable v = net.GetVariable (j);
    Domain d = v.GetDomain ();
    
    int j_domain_size = d.GetSize ();

    for (int a = 0; a < j_domain_size; a++)
      if ( (net.GetConstraint (j, j).GetConstraint (a, a))
	   && (domains[j][a] == -1) ) {
	domain_count++;
	NotifyConsistencyCheck ();
	if (net.GetConstraint (current, j).GetConstraint (value, a) == false) {
	  domains[j][a] = current;
	  delete_count++;
	}
      }
    
    if (delete_count > 0) {
      if (domain_count > delete_count) {	//	there's still domain values left
	Push (j);
      }
      
      checking[current][j] = true;
      
      NotifyDomainRestriction (current, j, domains[j], domain_count, delete_count);
    }
    
    return domain_count - delete_count;
  }
	
  /*
   *	Function to Restore the domain of variables which were eliminated due to the
   *	instanciation of variable "i". A variable is known to have had a
   *	value deleted by looking at the "checking" matrix. [CSPLib]
   */
  
  private void Restore (int i) {
    int restore_count = 0;
    int k;
    
    for (int j = i + 1; j < size; j++)
      if (checking[ordering[i]][ordering[j]]) {
	checking [ordering[i]][ordering[j]] = false;
	
	k = net.GetVariable (ordering[j]).GetDomain().GetSize();
	restore_count = 0;
	for (int l = 0; l < k; l++) {
	  if (domains[ordering[j]][l] == ordering [i]) {
	    domains[ordering[j]][l] = -1;
	    restore_count++;
	  }
	}
	
	NotifyDomainRestoration (ordering[i], ordering[j], domains[ordering[j]], restore_count);
      }
  }
  
  /*
   *	Clear the setup data structures.
   */
  
  private void ClearSetup (int n, int k) {
    
    int i, j;
    
    InitStack (n);
    
    for (i = 0; i < n; i ++) {
      for (j = 0; j < k; j ++)
	domains[i][j] = -1;
      for (j = 0; j < n; j ++)
	checking[i][j] = false;
      ordering [i] = i;
    }
    CalculateConnections ();
  }
 
  /*
   *    Find the variable with the minimum label from current position. 
   *    And build the ordering vector.
   */
  
  private void FindNextVar (int pos) {
    
    int var = 0;
    int minDomain = MAX_DOMAIN;
    int dom;
    int posOld = 0;
    
    int numTieBreak = -1;
    
    for (int i=0; i < size; i ++){
      tieBreak [i] = -1;
    }
    for (int i = pos; i < size; i ++) {
      
      
      dom = CalculateDomain (i);
      if (dom < minDomain) {
	minDomain = dom;
	var = ordering[i];
	posOld = i;
	
	/* Initialize the tie break structure */
	
	numTieBreak = 0;
	tieBreak[numTieBreak] = ordering[i];
	
      } else if (dom == minDomain) {
	
	/* Tie Break */
	numTieBreak ++;
	tieBreak[numTieBreak] = ordering[i];
      }
    }
    
    if (numTieBreak >= 1) {
      /* TIE BREAK */
      
      int minConnected = MAX_CONSTRAINTS;
      int connected;
      
      for (int k=0; k <= numTieBreak; k ++) {
	connected = connections [tieBreak[k]];
	if (connected < minConnected) {
	  minConnected = connected;
	  posOld = Position(tieBreak[k]);
	  var = ordering[Position(tieBreak[k])];
          
	}
      }
    }
    
    /* There is ordering */
    if (pos != posOld) {
      //System.out.println ("ORDER : " + pos + " FOR " + posOld);
      int tmp = ordering[pos];
      ordering [pos] = var;
      ordering [posOld] = tmp; 
    }  
  }
  
  /*
   *    Find and return the positon of the variable var
   */
  
  private int Position (int var) {
    
    int pos = 0;
    boolean found = false;
    
    for (int i = 0; ((i < size) && (!found)); i ++) {    
      if (var == ordering [i]) {
        pos = i;
        found = true;
      }      
    }
    return pos;
  }
  
  private int CalculateDomain (int pos) {
    int res = 0;
    
    int d_size = net.GetVariable (ordering[pos]).GetDomain().GetSize ();
    
    for (int i=0; i < d_size; i++) {
      if (domains [ordering[pos]][i] == -1) {
        res ++;
      }
    }
    return res;
  }
  
  private void CalculateConnections () {
             
    int res = 0;
    
    for (int j = 0; j < size; j ++) {
      Variable v1 = net.GetVariable (j);
      
      for (int i = 0; i < size; i ++) {
        if (j != i) {
          
          /* It's not the same variable */
          
          Variable v2 = net.GetVariable (i);
          
          int d1_size = v1.GetDomain().GetSize();
          int d2_size = v2.GetDomain().GetSize();
          
          for (int k1=0; k1 < d1_size; k1++) {
            for (int k2=0; k2 < d2_size; k2 ++) {
              if ((domains[j][k1] == -1) 
                  && (domains[i][k2] == -1)) {
                if (net.GetConstraint (v1.GetName (),v2.GetName ()).GetConstraint (k1, k2)) {
		  res ++;
                }
              }
            }
          }
        }
      }
      connections [j] = res;
    }
  }

  /*
   *	Stack functions for arc consistency where only variables have to be stored.
   *	Push pushes a variable on the stack and marks it as being on the stack,	
   *	unless the variable was already on the stack, in wich case it is not added
   *	again. Pop gets the top value off the stack. The stack_empty function 
   *	returns whether the stack is empty or not. [CSPLib]
   */
  
  private void Push (int a) {
    if (!on_stack[a]) {
      top ++;
      stack[top] = a;
      on_stack [a] = true;
    }
  }
  
  private int Pop () {
    int a = stack[top];
    on_stack[a] = false;
    top --;
    return a;
  }
  
  private void InitStack (int n) {
    top = 0;
    for (int i = 0; i < n; i ++)
      on_stack[i] = false;
  }
  
  private boolean StackEmpty () {
    return (top == 0);
  }
}
