package JCL;

import JCL.Solver;

/**
 * Implementation of a Graph-Based BackJumping
 * solver for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version 19-11-96
 */

public class GBJSolver extends Solver {
	
	final static int MAX_DOMAIN = 30;

	boolean parents[][];	//	function returns the parents set of Xi
	boolean P[];		//	P global set variable wich contains variables
				//	that may have caused the inconsistency

        int count = 0;

	final static String NAME = "Graph-Based Backjumping";
	
	/*
	 *	Constructors.
	 */

	public GBJSolver () {
	}

	public GBJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		
		//	Initialize variables

		parents = new boolean[size][size];
		P = new boolean[size];

		ClearSetup (size, MAX_DOMAIN);
		
		//	Start recursion

		Domain d = net.GetVariable (0).GetDomain ();

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}
	
	/*
 	 *	Solve the constraint network using the Dechter's GRAPH_BASED BACKJUMPING
 	 *	(GBJ) method.  If this is the first variable then initialize the data
 	 *	structures.  If a solution is found then update the "found" variable, call
 	 *	the function to process the solution, and return the value according to
 	 *	whether the first solution is desired or all solutions.  Then, check if the
 	 *	timer has expired, and if it has, return immediately.  Otherwise, begin
 	 *	checking each possible instantiation of the variable.  For each domain
 	 *	value, perform the following steps.  First, if preprocessing eliminated the
 	 *	value, disregard the rest of the loop.  Otherwise, instantiate the variable,
 	 *	check if the network is still consistent, and then call the backtracking
 	 *	routine recursively.  After checking all possible domain values, determine
 	 *	the variable to jump back to and return this variable. [CSPLib]
 	 */

	
	private int RecursiveSolve (int current, int k) {

		//	Local Variables
		int kk = 0;
		Variable v;
		
		int curr = count;

		NotifyEnterLevel (current);

		if (current >= size) {

			//	A solution has been found
			
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				count ++;
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		int jump;

		for (int i = 0; i < k; i++) {
			if (net.GetConstraint (current, current).GetConstraint (i, i) == false) {
				System.err.println ("CBJ : variable not consistent with itself !");
				continue;
			}

			indexes[current] = i;
			NotifyInstanciation (current, i);

			if (IsConsistent (current)) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				jump = RecursiveSolve (current + 1, kk);
				if (jump != current) {
					NotifyLeaveLevel (current);
					return jump;
				}
			}
		}
		

		jump = (curr == count) ? UnionParents (current) : current - 1;

		NotifyLeaveLevel (current);

		return jump;
	}


	/*
 	 *	Check if the current instantiation of the variables is consistent by
 	 *	looking at the edge between the current variable and all previous variables.
	 *	[CSLib]
 	 */

	private boolean IsConsistent (int current) {
		for (int i = 0; i < current; i++) {
			NotifyConsistencyCheck ();

			if (net.GetConstraint (current, i)
				.GetConstraint (indexes[current], indexes[i]) == false) {
	
				return false;
			}
		}
		return true;
	}

	/*
 	 *	Return whether there exists some constraint between two variables, which is
 	 *	simple searching an edge for a "0" entry. [CSPLib]
 	 */

	private boolean Constraint(int i, int j) {
    
		int a, b;
		int dom1, dom2;

		if (net.GetConstraint(i, j).IsExplicit ()) {

			dom1 = net.GetVariable (i).GetDomain ().GetSize ();
			dom2 = net.GetVariable (j).GetDomain ().GetSize ();

			for (a = 0; a < dom1; a++)
				for (b = 0; b < dom2; b++) {
					if (net.GetConstraint (i, j) .GetConstraint(a,b) == false)
							return(true);
				}
			return(false);
		} else return false;
	}


	/*
 	 * 	Clear the setup data structures.  Then, compute the parent tree and store
 	 * 	this information in the data structure to be used when backjumping is to be
 	 * 	done.  See the "Dechter" paper for a full description of the data structure.
	 * 	[CSPLib]
 	 */

	private void ClearSetup (int n, int k) {
    
		int i, l;
		int j;

		for (i = 0; i < n; i++) {
		P[i] = false; 
			for (j = 0; j < n; j++)
         	   	parents[i][j] = false;
    		}

		for (i = 0; i < n; i++)
			for (j = 0; j < i; j++)
				if (Constraint(i, j)) {
					for (l = 0; l < j; l++)
                    				parents[i][l] = parents[i][l] || parents[j][l];
					parents[i][j] = true;
					parents[i][i] = true;  // = j in the CSPLib
				}
	}


	/*
 	 *	Create the union of two parent sets.  This is necessary to be able to
 	 *	continue graph-based backjumping correctly to ensure the correct variable
 	 *	will not be missed. [CSPLib]
 	*/

	private int UnionParents (int i) {

		int j, temp = 0;

		for (j = 0; j < i; j++) {
			P[j] = P[j] || parents[i][j];
			if (P[j])
				temp = j;  // return max (P)
    		}
		P[i] = false;	// delete (h, P)
		return(temp);
	}
}
