package JCL;

import JCL.Solver;

/**
 * Implementation of a Forward Checking with Graph-Based BackJumping
 * solver for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last Update 14-11-96
 */

public class FC_GBJSolver extends Solver {
	
	final static int MAX_DOMAIN = 30;
	
	/*
	 *	Data structures to hold the required information.  See the accompanying
 	 *	research articles for complete descriptions of the structures.  The
 	 *	"domains" structure works as follows.  If an entry contains a 0, then the
  	 *	value is still allowed.  Otherwise, it contains the variable whose
	 *	instantiation eliminated it.  Note that this data structure has no
	 *	information from preprocessing (which is kept directly in the "C" matrix).
	 */

	int domains[][];
	boolean checking[][];
	boolean parents[][];	//	function returns the parents set of Xi
	boolean P[];		//	P global set variable wich contains variables
				//	that may have caused the inconsistency
	int mcl[][];		//	N, K
	int mbl[];		//	N

        int count = 0;

	final static String NAME = "Forward Checking & Graph-Based Backjumping hibrid";
	
	/*
	 *	Constructors.
	 */

	public FC_GBJSolver () {
	}

	public FC_GBJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		
		//	Initialize variables

		parents = new boolean[size][size];
		P = new boolean[size];
		domains = new int [size][MAX_DOMAIN];
		checking = new boolean[size][size];
		
		ClearSetup (size, MAX_DOMAIN);
		
		//	Start recursion

		Domain d = net.GetVariable (0).GetDomain ();

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}


	/*
	 *	Solve the constraint network using the FORWARD CHECKING with Dechter's
	 *	GRAPH_BASED BACKJUMPING (FC-GBJ) method.  If this is the first variable
	 *	then initialize the data structures.  If a solution is found then update
	 *	the "found" variable, call the function to process the solution, and return
	 *	the value according to whether the first solution is desired or all
	 *	solutions.  Then, check if the timer has expired, and if it has, return
	 *	immediately.  Otherwise, begin checking each possible instantiation of the
	 *	variable.  For each domain value, perform the following steps.  First, if
	 *	preprocessing eliminated the value, disregard the rest of the loop.
	 *	Otherwise, instantiate the variable, check if the network is still
	 *	consistent, and then call the backtracking routine recursively.  After
	 *	checking each domain value, restore the domains which were eliminated by the
	 *	instantiation.  After checking all possible domain values, determine the
	 *	variable to jump back to, update the data structures, and return.
	 */
	
	private int RecursiveSolve (int current, int k) {

		//	Local Variables
		int kk = 0;
		Variable v;

		int curr = count;

		NotifyEnterLevel (current);

		if (current >= size) {

			//	A solution has been found
			
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				count ++;
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		int jump;

		for (int i = 0; i < k; i++) {

			if ( (net.GetConstraint (current, current).GetConstraint (i, i) == false) 
				|| (domains[current][i] != -1)) {
				continue;
			}

			indexes[current] = i;
			NotifyInstanciation (current, i);

			if (IsConsistent (current, indexes[current])) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				jump = RecursiveSolve (current + 1,kk);
				if (jump != current) {
					NotifyLeaveLevel (current);
					return jump;
				}
			}
			Restore (current);
		}
		

		jump = (curr == count) ? UnionParents (current) : current - 1;
		for (int i = current; i > jump; i--)
			Restore (i);

		NotifyLeaveLevel (current);

		return jump;
	}

	/*
	 *	Check if the current instantiation of the variables is consistent by
	 *	checking if the current instantiation of a variable will reduce some
	 *	future variable's domain to the empty set, in which case the
	 *	appropriate result is returned. [CSPLib]
	 */

	private boolean IsConsistent (int current, int value) {

		for (int i = current + 1; i < size; i++) {
			if (CheckForward (current, i, value) == 0) {
				UpdateParents (current, i);
				return false;
			}
		}

		return true;
	}
	/*
 	 *	Return whether there exists some constraint between two variables, which is
 	 *	simple searching an edge for a "0" entry. [CSPLib]
 	 */

	private boolean Constraint(int i, int j) {
    
		int a, b;
		int dom1, dom2;

		if (net.GetConstraint (i, j).IsExplicit ()) {

			dom1 = net.GetVariable (i).GetDomain ().GetSize ();
			dom2 = net.GetVariable (j).GetDomain ().GetSize ();

			for (a = 0; a < dom1; a++)
				for (b = 0; b < dom2; b++) {
					if (net.GetConstraint (i,j).GetConstraint (a, b) == false)
							return(true);
				}
			return(false);
		} else return false;
	}



	/*
 	 * 	Clear the setup data structures.  Then, compute the parent tree and store
 	 * 	this information in the data structure to be used when backjumping is to be
 	 * 	done.  See the "Dechter" paper for a full description of the data structure.
	 * 	[CSPLib]
 	 */

	private void ClearSetup (int n, int k) {
    
		int i, l;
		int j;

		for (i = 0; i < n; i++) {

			P[i] = false; 

			for (j = 0; j < n; j++) {
         	   		parents[i][j] = false;
				checking[i][j] = false;
			}
			for (j = 0; j < MAX_DOMAIN; j++)
				domains[i][j] = -1;
    		}

		for (i = 0; i < n; i++)
			for (j = 0; j < i; j++)
				if (Constraint(i, j)) {
					for (l = 0; l < j; l++)
                    				parents[i][l] = parents[i][l] || parents[j][l];
					parents[i][j] = true;
					parents[i][i] = true;  // = j in the CSPLib
				}
	}

	/*
	 *	Update the variable to jump back to by looking at the lists of parents. As
	 *	described by Dechter, this is necessary to be able to continue graph-based
	 *	backjumping correctly to ensure the corrent variable will not be missed. 
	 *	[CSPLib]
	 */

	private void UpdateParents (int i, int j) {

		int k;

		for (k = 0; k < i; k++)
			P[k] = P[k] || parents[j][k];
	}




	/*
 	 *	Create the union of two parent sets.  This is necessary to be able to
 	 *	continue graph-based backjumping correctly to ensure the correct variable
 	 *	will not be missed. [CSPLib]
	 */

	private int UnionParents (int i) {

		int j, temp = 0;

		for (j = 0; j < i; j++) {
			P[j] = P[j] || parents[i][j];
			if (P[j])
				temp = j;  // return max (P)
    		}
		P[i] = false;	// delete (h, P)
		return(temp);
	}


	/*
	 *	This function checks the edge between variables "i" and "j" to see
	 *	which values in the domain of "j" can be eliminated due to the
	 *	current instantiation of "i" as "value". Note that given a variable
	 *	"i" and an instantiation "v", domains[i][v] is -1 if the value is
	 *	still allowed, otherwise it contains the variable which caused it
	 *	to be eliminated. If any value in "j" is deleted, this is noted in
	 *	the "checking" matrix. The number of values in the domain of "j"
	 *	after all deletions is returned. [CSPLib]
	 */

	private int CheckForward (int i, int j, int value) {
		int old_count = 0;
		int delete_count = 0;

		Variable v = net.GetVariable (j);
		Domain d = v.GetDomain ();

		int j_domain_size = d.GetSize ();

		for (int n = 0; n < j_domain_size; n++)
			if ( (net.GetConstraint (j, j).GetConstraint (n, n))
			  && (domains[j][n] == -1) ) {
				old_count++;
				NotifyConsistencyCheck ();
				if (net.GetConstraint (i, j).GetConstraint (value, n) == false) {
					domains[j][n] = i;
					delete_count++;
				}
			}

		if (delete_count > 0) {
			checking[i][j] = true;
			NotifyDomainRestriction (i, j, domains[j], old_count, delete_count);
		}

		return old_count - delete_count;
	}
	
	/*
	 *	Restore the domain of variables which were eliminated due to the
	 *	instanciation of variable "i". A variable is known to have had a
	 *	value deleted by looking at the "checking" matrix. [CSPLib]
	 */

	private void Restore (int i) {
		int restore_count = 0;

		for (int j = i + 1; j < size; j++)
			if (checking[i][j]) {
				checking[i][j] = false;
				for (int l = 0; l < MAX_DOMAIN; l++) {
					if (domains[j][l] == i) {
						domains[j][l] = -1;
						restore_count++;
					}
				}

				NotifyDomainRestoration (i, j, domains[j], restore_count);
			}
	}

}
