package JCL;

/**
 * Implementation of a simple Forward Checking with DVO solver for the Java
 * Constraint Library.
 *
 * @author Marc Torrens
 * @version Last update 14-04-97
 */

public class FC_DVOSolver extends Solver {
  
  int domains[][];
  boolean checking[][];
  
  int ordering[];           /* vector to store the ordering sequence of variables.*/
  
  int connections[];

  int tieBreak [];

  int current_domain_size;
  
  final static String NAME = "Simple Forward Checking with Dynamic Variable Ordering";
  final static int MAX_DOMAIN = 100000;
  final static int MAX_CONSTRAINTS = 100000;
  
  /*
   *	Constructors.
   */
  
  public FC_DVOSolver () {
  }
  
  public FC_DVOSolver (Network net) {
    SetNetwork (net);
  }
  
  /**
    *	Return the name of the algorithm.
    */
  
  public String GetName () {
    return new String (NAME);
  }
  
  /**
    *     Try to solve the problem.
    */
  
  public void Solve () {
    
     //	Initialize variables
    
    domains = new int[size][MAX_DOMAIN];
    checking = new boolean[size][size];
    ordering = new int [size];
    tieBreak = new int [size];
    connections = new int [size];

    ClearSetup (size, MAX_DOMAIN);
    
    
    //	Start recursion
    
    FindNextVar (0);
    int firstVar = ordering [0];

    Domain d = net.GetVariable (firstVar).GetDomain ();
    
    try {
      
      RecursiveSolve (0, d.GetSize());
      
    } catch (SolutionFoundException e) {
    }
  }
  
  /*
   *	Solve the constraint network using the simple FORWARD CHECKING (FC) method.
   *	If a solution is found then notify it and return the value according to
   *	whether the first solution is desired or all solutions. Otherwise, begin
   *	checking each ossible nstantiation of the variable.  For each domain value,
   *	perform the ollowing steps. First, if preprocessing eliminated the value,
   *	disregard the rest of the loop.  Otherwise, instantiate the variable,
   *	check if the network is till consistent, and then call the backtracking
   *	routine recursively. After checking each domain value, restore the domains
   *	which were eliminated by the instantiation. After checking all possible
   *	domain values, return. [CSPLib]
   */
  
  private boolean RecursiveSolve (int current, int k) {
    
    //	 Local Variables
    int kk = 0;
    Variable v;
    
    NotifyEnterLevel (current);
    
    if (current >= size) {
      
      //	A solution has been found
      
      NotifySolution ();
      
      if (FindMoreSolutions ()) {
	NotifyLeaveLevel (current);
	return false;
      } else
	throw new SolutionFoundException ();
    }
    
    for (int i = 0; i < k; i++) {
      if ( (net.GetConstraint (ordering[current], ordering[current]).GetConstraint (i, i) == false)
	   || (domains[ordering[current]][i] != -1) ) {
	continue;
      }
      
      indexes[ordering[current]] = i;
      NotifyInstanciation (ordering[current], i);
      
      if (IsConsistent (current, indexes[ordering[current]])) {
	
	if (current < (size - 1)) {
	  
	  FindNextVar(current+1);
	  v = net.GetVariable (ordering[current +1]);
	  kk = v.GetDomain ().GetSize();
	} 
	
	if (RecursiveSolve (current + 1, kk)) {
	  NotifyLeaveLevel (current);
	  return true;
	}
      }
      
      Restore (current);
    }
    
    NotifyLeaveLevel (current);
    return false;
  }
  
  /*
   *	Check if the current instantiation of the variables is consistent by
   *	checking if the current instantiation of a variable will reduce some
   *	future variable's domain to the empty set, in which case the
   *	appropriate result is returned. [CSPLib]
   */
  
  private boolean IsConsistent (int current, int value) {
    for (int i = current + 1; i < size; i++) {
      if (CheckForward (ordering[current], ordering[i], value) == 0)
	return false;
    }
    return true;
  }
  
  /*
   *	This function checks the edge between variables "i" and "j" to see
   *	which values in the domain of "j" can be eliminated due to the
   *	current instantiation of "i" as "value". Note that given a variable
   *	"i" and an instantiation "v", domains[i][v] is -1 if the value is
   *	still allowed, otherwise it contains the variable which caused it
   *	to be eliminated. If any value in "j" is deleted, this is noted in
   *	the "checking" matrix. The number of values in the domain of "j"
   *	after all deletions is returned. [CSPLib]
   */
  
  private int CheckForward (int i, int j, int value) {
    
    int old_count = 0;
    int delete_count = 0;
    
    Variable v = net.GetVariable (j);
    Domain d = v.GetDomain ();
    
    for (int n = 0; n < d.GetSize (); n++)
      if ( (net.GetConstraint (j, j).GetConstraint (n, n))
	   && (domains[j][n] == -1) ) {
	old_count++;
	NotifyConsistencyCheck ();
	if (net.GetConstraint (i, j).GetConstraint (value, n) == false) {
	  domains[j][n] = i;
	  delete_count++;
	}
      }
    
    if (delete_count > 0) {
      checking[i][j] = true;
      NotifyDomainRestriction (i, j, domains[j], old_count, delete_count);
    }
    return old_count - delete_count;
  }
  
  /*
   *	Restore the domain of variables which were eliminated due to the
   *	instanciation of variable "i". A variable is known to have had a
   *	value deleted by looking at the "checking" matrix. [CSPLib]
   */
  
  private void Restore (int i) {
    int restore_count = 0;
    
    for (int j = i + 1; j < size; j++)
      if (checking[ordering[i]][ordering[j]]) {
	checking[ordering[i]][ordering[j]] = false;
	
	Variable v = net.GetVariable(ordering[j]);
	Domain d = v.GetDomain ();
	
	restore_count = 0;
	for (int l = 0; l < d.GetSize (); l++) {
	  if (domains[ordering[j]][l] == ordering[i]) {
	    domains[ordering[j]][l] = -1;
	    restore_count++;
	  }
	}
	
	NotifyDomainRestoration (i, j, domains[ordering[j]], restore_count);
      }
  }
  
  /*
   *	Clear the setup data structures.
   */
  
  private void ClearSetup (int n, int k) {
    int i, j;
    
    for (i = 0; i < n; i ++) {
      
      Variable v = net.GetVariable (i);
      Domain d = v.GetDomain ();
      int d_size = d.GetSize ();
      
      for (j = 0; j < d_size; j ++)
	domains[i][j] = -1;
      for (j = 0; j < n; j ++)
	checking[i][j] = false;
      ordering [i] = i;
    }
    CalculateConnections ();
    
  }

  /*
   *	Find the variable with the minimum label from current position. 
   *    And build the ordering vector.
   */
  
  private void FindNextVar (int pos) {
   
    int var = 0;
    int minDomain = MAX_DOMAIN;
    int dom;
    int posOld = 0;
   
    int numTieBreak = -1;
     
     for (int i=0; i < size; i ++){
       tieBreak [i] = -1;
     }
     for (int i = pos; i < size; i ++) {
       
       
       dom = CalculateDomain (i);
       if (dom < minDomain) {
	 minDomain = dom;
	 var = ordering[i];
	 posOld = i;
	 
	 /* Initialize the tie break structure */
	
	 numTieBreak = 0;
	 tieBreak[numTieBreak] = ordering[i];
	 
	 } else if (dom == minDomain) {
	 
	   /* Tie Break */
	   numTieBreak ++;
	   tieBreak[numTieBreak] = ordering[i];
	 }
     }
     
     if (numTieBreak >= 1) {
       /* TIE BREAK */
     
       int minConnected = MAX_CONSTRAINTS;
       int connected;

       for (int k=0; k <= numTieBreak; k ++) {
 	 connected = connections [tieBreak[k]];
 	 if (connected < minConnected) {
 	   minConnected = connected;
	   posOld = Position(tieBreak[k]);
	   var = ordering[Position(tieBreak[k])];
	  
 	 }
       }
       
     }
     
     /* There is ordering */
     if (pos != posOld) {
       //System.out.println ("ORDER : " + pos + " FOR " + posOld);
       int tmp = ordering[pos];
       ordering [pos] = var;
       ordering [posOld] = tmp;
      
     }
     
  }
  
  /*
   *    Find and return the positon of the variable var
   */
  
  private int Position (int var) {
    
    int pos = 0;
    boolean found = false;
    
    for (int i = 0; ((i < size) && (!found)); i ++) {
      
      if (var == ordering [i]) {
 	pos = i;
	found = true;
      }      
    }
    return pos;
  }
  
  private int CalculateDomain (int pos) {
    int res = 0;
    
    int d_size = net.GetVariable (ordering[pos]).GetDomain().GetSize ();

    for (int i=0; i < d_size; i++) {
      if (domains [ordering[pos]][i] == -1) {
	res ++;
      }
    }
    return res;
  }

  private void CalculateConnections () {

    int res = 0;
    
    for (int j = 0; j < size; j ++) {
      Variable v1 = net.GetVariable (j);

      for (int i = 0; i < size; i ++) {
	if (j != i) {
	  
	  /* It's not the same variable */
	  
	  Variable v2 = net.GetVariable (i);
	  
	  int d1_size = v1.GetDomain().GetSize();
	  int d2_size = v2.GetDomain().GetSize();
	  
	  for (int k1=0; k1 < d1_size; k1++) {
	    for (int k2=0; k2 < d2_size; k2 ++) {
	      if ((domains[j][k1] == -1) 
		  && (domains[i][k2] == -1)) {
		if (net.GetConstraint (v1.GetName (),v2.GetName ()).GetConstraint (k1, k2)) {
		  res ++;
		}
	      }
	    }
	  }
	}
      }	
      connections [j] = res;
    }
  }
}


