package JCL;

/**
 * Implementation of a Forward Checking & Conflict-Directed Backjumping solver
 * for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last update 14-11-96
 */

public class FC_CBJSolver extends Solver {

	int domains[][];
	int conflicts[][];
	boolean checking[][];
	int count = 0;

	int current_domain_size;

	final static String NAME = "Forward Checking & Conflict-Directed Backjumping";
	
	/*
	 *	Constructors.
	 */

	public FC_CBJSolver () {
	}

	public FC_CBJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		
		//	Initialize variables
		
		domains = new int[size][MAX_DOMAIN];
		checking = new boolean[size][size];
		conflicts = new int[size][size];
		
		ClearSetup (size, MAX_DOMAIN);

		//	Start recursion

		Domain d = net.GetVariable (0).GetDomain ();		

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}
	
	/*
 	 *	Solve the constraint network using the FORWARD CHECKING with CONFLICT-DIRECTED
 	 *	BACKJUMPING (FC-CBJ) method.  If this is the first variable then initialize
 	 *	the data structures.  If a solution is found then update the "found"
 	 *	variable, call the function to process the solution, and return the value
 	 *	according to whether the first solution is desired or all solutions.  Then,
 	 *	check if the timer has expired, and if it has, return immediately.
 	 *	Otherwise, begin checking each possible instantiation of the variable.  For
 	 *	each domain value, perform the following steps.  First, if preprocessing
 	 *	eliminated the value, disregard the rest of the loop.  Otherwise,
 	 *	instantiate the variable, check if the network is still consistent, and then
 	 *	call the backtracking routine recursively.  After checking each domain
 	 *	value, restore the domains which were eliminated by the instantiation.
 	 *	After checking all possible domain values, determine the variable to jump
 	 *	back to, update the data structures, and return.
 	 */
	
	private int RecursiveSolve (int current, int k) {

		int fail;
		int jump = 0;
		int i;

		int curr = count;


		//	Local Variables
		int kk = 0;
		Variable v;

		NotifyEnterLevel (current);

		if (current >= size) {

			//	A solution has been found

			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				count ++;
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		for (i = 0; i < k; i++) {
			if ( (net.GetConstraint (current, current).GetConstraint (i, i) == false)
			  || (domains[current][i] != -1) )
				continue;

			indexes[current] = i;
			NotifyInstanciation (current, i);

			fail = IsConsistent (current, indexes[current]);

			if (fail == 0) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				jump = RecursiveSolve (current + 1, kk);
				if (jump != current)
					return jump;	
			}
			if (fail != 0)
				UnionChecking (current, fail);

			Restore (current);
		}

		NotifyLeaveLevel (current);
		if (curr == count) {
			jump = 0;
			for (i = 0; i < current; i++)
				if (conflicts[current][i] != -1)
					jump = i;
			for (i = jump + 1; i < current; i++)
				if (checking[i][current])
					jump = i; 
		} else
			jump = current - 1;

		UnionChecking (current, current);
		UnionConflicts (jump, current);
		for (i = current; i > jump; i--) {
			EmptyConflicts (i);
			Restore (i);
		}
		if (current != 0)
		  Restore (jump);
		return jump;
	}

	/*
	 *	Check if the current instantiation of the variables is consistent by
	 *	checking if the current instantiation of a variable will reduce some
	 *	future variable's domain to the empty set, in which case the
	 *	appropriate result is returned. [CSPLib]
	 */

	private int IsConsistent (int current, int value) {
 
		for (int i = current + 1; i < size; i++) {
			if (CheckForward (current, i, value) == 0)
{
				return i;
			}
		}

		return 0;
	}

	/*
	 *	This function checks the edge between variables "i" and "j" to see
	 *	which values in the domain of "j" can be eliminated due to the
	 *	current instantiation of "i" as "value". Note that given a variable
	 *	"i" and an instantiation "v", domains[i][v] is -1 if the value is
	 *	still allowed, otherwise it contains the variable which caused it
	 *	to be eliminated. If any value in "j" is deleted, this is noted in
	 *	the "checking" matrix. The number of values in the domain of "j"
	 *	after all deletions is returned. [CSPLib]
	 */

	private int CheckForward (int i, int j, int value) {
		int old_count = 0;
		int delete_count = 0;

		Variable v = net.GetVariable (j);
		Domain d = v.GetDomain ();

		int j_domain_size = d.GetSize ();

		for (int n = 0; n < j_domain_size; n++)
			if ( (net.GetConstraint (j, j).GetConstraint (n, n))
			  && (domains[j][n] == -1) ) {
				old_count++;
				NotifyConsistencyCheck ();
				if (net.GetConstraint (i, j).GetConstraint (value, n) == false) {
					domains[j][n] = i;
					delete_count++;
				}
			}

		if (delete_count > 0) {
			checking[i][j] = true;
			NotifyDomainRestriction (i, j, domains[j], old_count, delete_count);
		}

		return old_count - delete_count;
	}
	
	/*
	 *	Restore the domain of variables which were eliminated due to the
	 *	instanciation of variable "i". A variable is known to have had a
	 *	value deleted by looking at the "checking" matrix. [CSPLib]
	 */

	private void Restore (int i) {

		int restore_count = 0;
		int domain;

		for (int j = i + 1; j < size; j++)
			if (checking[i][j]) {
				checking[i][j] = false;
				domain = net.GetVariable (j).GetDomain().GetSize();
				for (int l = 0; l < domain; l++) {
					if (domains[j][l] == i) {
						domains[j][l] = -1;
						restore_count++;
					}
				}

				NotifyDomainRestoration (i, j, domains[j], restore_count);
			}
	}

	/*
	 *	Clear the setup data structures.
	 */

	private void ClearSetup (int n, int k) {

		for (int i = 0; i < size; i++) {
			for (int j = 0; j < MAX_DOMAIN; j++)
				domains[i][j] = -1;
			for (int j = 0; j < size; j++) {
				checking[i][j] = false;
				conflicts[i][j] = -1;
			}
		}
	}

	/*
	 *	Create the union of two sets which hold where the conflicts took place.  As
 	 *	described by Prosser, this is necessary to be able to continue conflict-directed
 	 *	backjumping correctly to ensure the correct variable will not be missed. 
	 *	[CSPLib]
 	 */

	private void UnionChecking(int i, int j) {
    	
		int n;

    		for (n = 0; n < i; n++)
			conflicts[i][n] = ((conflicts[i][n] > -1) || (checking[n][j])) ? 0 : -1;
	}
	
	/*
	 *	Create the union of two sets which hold where the conflicts took place.
	 *	As described by Prosser, this is necessary to be able to continue
	 *	conflict-directed backjumping correctly to ensure the correct variable
	 *	will not be missed. The maximum in the set for variable "i" is retained
	 *	in position "conflicts[i][i]" for future reference. [CSPLib]
	 */

	private void UnionConflicts (int i, int j) {
                for (int n = 0; n < i; n++) {
                        conflicts[i][n]
                                = ((conflicts[i][n] > -1) || (conflicts[j][n] > -1)) ? 0 : -1;

                        if ((conflicts[i][n] > -1) && (conflicts[i][i] < n)) 
                                conflicts[i][i] = n;
                }
        }
	/*
	 *	Reset the sets between the given indexes to be empty. This allows
	 *	for new information to be stored in these sets without the old
	 *	information misleading the algorithm. [CSPLib]
	 */
	
	private void EmptyConflicts (int i) {
		for (int j = 0; j <= i; j++)
			conflicts[i][j] = -1;
	}

}
