package JCL;

/**
 * Implementation of a simple Forward Checking solver for the Java
 * Constraint Library.
 *
 * @author Erik Bruchez
 * @version Last update 14-11-96
 */

public class FCSolver extends Solver {

  int domains[][];
  boolean checking[][];
  
  int current_domain_size;
  
  final static String NAME = "Simple Forward Checking";
  
  /*
   *	Constructors.
   */
  
  public FCSolver () {
  }
  
  public FCSolver (Network net) {
    SetNetwork (net);
  }

  /**
    *	Return the name of the algorithm.
    */

  public String GetName () {
    return new String (NAME);
  }
  
  /**
    *	Try to solve the problem.
    */
  
  public void Solve () {
    
    //	Initialize variables
    
    domains = new int[size][MAX_DOMAIN];
    checking = new boolean[size][size];
    
    ClearSetup (size, MAX_DOMAIN);
    
    
    //	Start recursion
    
    Domain d = net.GetVariable (0).GetDomain ();
    
    try {
      
      RecursiveSolve (0, d.GetSize());
      
    } catch (SolutionFoundException e) {
    }
  }
  
  /*
   *	Solve the constraint network using the simple FORWARD CHECKING (FC) method.
   *	If a solution is found then notify it and return the value according to
   *	whether the first solution is desired or all solutions. Otherwise, begin
   *	checking each ossible nstantiation of the variable.  For each domain value,
   *	perform the ollowing steps. First, if preprocessing eliminated the value,
   *	disregard the rest of the loop.  Otherwise, instantiate the variable,
   *	check if the network is till consistent, and then call the backtracking
   *	routine recursively. After checking each domain value, restore the domains
   *	which were eliminated by the instantiation. After checking all possible
   *	domain values, return. [CSPLib]
   */
  
  private boolean RecursiveSolve (int current, int k) {
    
    //	 Local Variables
    int kk = 0;
    Variable v;
    
    NotifyEnterLevel (current);
    
    if (current >= size) {
      
      //	A solution has been found
      
      NotifySolution ();
      
      if (FindMoreSolutions ()) {
	NotifyLeaveLevel (current);
	return false;
      } else
	throw new SolutionFoundException ();
    }
    
    for (int i = 0; i < k; i++) {
      if ( (net.GetConstraint (current, current).GetConstraint (i, i) == false)
	   || (domains[current][i] != -1) ) {
	continue;
      }
      
      indexes[current] = i;
      NotifyInstanciation (current, i);
      
      if (IsConsistent (current, indexes[current])) {
	
	if (current < (size - 1)) {
	  
	  v = net.GetVariable (current +1);
	  kk = v.GetDomain ().GetSize();
	}
	
	if (RecursiveSolve (current + 1, kk)) {
	  NotifyLeaveLevel (current);
	  return true;
	}
      }
      
      Restore (current);
    }
    
    NotifyLeaveLevel (current);
    return false;
  }
  
  /*
   *	Check if the current instantiation of the variables is consistent by
   *	checking if the current instantiation of a variable will reduce some
   *	future variable's domain to the empty set, in which case the
   *	appropriate result is returned. [CSPLib]
   */
  
  private boolean IsConsistent (int current, int value) {
    for (int i = current+1; i < size; i++) {
      if (CheckForward (current, i, value) == 0)
	return false;
    }
    return true;
  }
  
  /*
   *	This function checks the edge between variables "i" and "j" to see
   *	which values in the domain of "j" can be eliminated due to the
   *	current instantiation of "i" as "value". Note that given a variable
   *	"i" and an instantiation "v", domains[i][v] is -1 if the value is
   *	still allowed, otherwise it contains the variable which caused it
   *	to be eliminated. If any value in "j" is deleted, this is noted in
   *	the "checking" matrix. The number of values in the domain of "j"
   *	after all deletions is returned. [CSPLib]
   */
  
  private int CheckForward (int i, int j, int value) {
    
    int old_count = 0;
    int delete_count = 0;
    
    Variable v = net.GetVariable (j);
    Domain d = v.GetDomain ();
    
    for (int n = 0; n < d.GetSize (); n++)
      if ( (net.GetConstraint (j, j).GetConstraint (n, n))
	   && (domains[j][n] == -1) ) {
	old_count++;
	NotifyConsistencyCheck ();
	if (net.GetConstraint (i, j).GetConstraint (value, n) == false) {
	  domains[j][n] = i;
	  delete_count++;
	}
      }
    
    if (delete_count > 0) {
      checking[i][j] = true;
      NotifyDomainRestriction (i, j, domains[j], old_count, delete_count);
    }
    return old_count - delete_count;
  }
  
  /*
   *	Restore the domain of variables which were eliminated due to the
   *	instanciation of variable "i". A variable is known to have had a
   *	value deleted by looking at the "checking" matrix. [CSPLib]
   */

  private void Restore (int i) {
    int restore_count = 0;
    
    for (int j = i + 1; j < size; j++)
      if (checking[i][j]) {
	checking[i][j] = false;
	
	Variable v = net.GetVariable(j);
	Domain d = v.GetDomain ();
	
	restore_count = 0;
	for (int l = 0; l < d.GetSize (); l++) {
	  if (domains[j][l] == i) {
	    domains[j][l] = -1;
	    restore_count++;
	  }
	}
	
	NotifyDomainRestoration (i, j, domains[j], restore_count);
      }
  }
  
  /*
   *	Clear the setup data structures.
   */
  
  private void ClearSetup (int n, int k) {
    int i, j;
    
    for (i = 0; i < n; i ++) {
      
      Variable v = net.GetVariable (i);
      Domain d = v.GetDomain ();
      int d_size = d.GetSize ();
      
      for (j = 0; j < d_size; j ++)
	domains[i][j] = -1;
      for (j = 0; j < n; j ++)
	checking[i][j] = false;
    }
    
  }
}
