package JCL;

import JCL.Solver;

/**
 * Implementation of a Constraint-Directed BackJumping
 * solver for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last update 19-11-96
 */

public class CBJSolver extends Solver {
	
	int conflicts[][];
	int count = 0 ;

	final static String NAME = "Constraint-Directed Backjumping";
	
	/*
	 *	Constructors.
	 */

	public CBJSolver () {
	}

	public CBJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {

		//	Initialize variables

		conflicts = new int[size][size];
	
		ClearSetup ();
		
		//	Start recursion
		
		Domain d = net.GetVariable (0).GetDomain ();

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}
	
	/*
	 *	Solve the constraint network using the CONFLICT-DIRECTED BACKJUMPING (CBJ) method.
	 *	If a solution is found then notify it and return the value according to whether
	 *	the first solution is desired or all solutions. Otherwise, begin checking each
	 *	possible instantiation of the variable. For each domain value, perform the
	 *	following steps. First, if preprocessing eliminated the value, disregard the rest
	 *	of the loop. Otherwise, instantiate the variable, check if the network is
	 *	still consistent, and then call the backtracking routine recursively. After
	 *	checking all possible domain values, determine the variable to jump back to
	 *	and return this value. [CSPLib]
	 */
	
	private int RecursiveSolve (int current, int k) {
	
		//	Local Variables
		int kk = 0;
		Variable v;
		
		int curr = count;
		
		NotifyEnterLevel (current);

		if (current == 0)
			ClearSetup ();		

		if (current >= size) {

			//	A solution has been found
			
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				count ++;
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		int jump;

		for (int i = 0; i < k; i++) {
			if (net.GetConstraint (current, current).GetConstraint (i, i) == false) {
				System.out.println ("CBJ : variable not consistent with itself !");
				continue;
			}

			indexes[current] = i;
			NotifyInstanciation (current, i);

			if (IsConsistent (current)) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				jump = RecursiveSolve (current + 1, kk);
				if (jump != current) {
//					NotifyLeaveLevel (current);
					return jump;
				}
			}
		}
	     
		jump = (curr == count) ? conflicts[current][current] : current - 1;
		UnionConflicts (jump, current);

		EmptyConflicts (jump, current);

//		NotifyLeaveLevel (current);
		for (int j = current; j > jump; j--) {
			NotifyLeaveLevelAux (j);
		}
      NotifyBackjump ();
		return jump;
	}

	/*
	 *	Check if the current instantiation of the variables is consistent by
	 *	looking at the edge between the current variable and all previous
	 *	variables. If a conflict exists, the set of conflicts with the current
	 *	variable is updated to reflect this fact. [CSPLib]
	 */

	private boolean IsConsistent (int current) {
		for (int i = 0; i < current; i++) {
			NotifyConsistencyCheck ();

			if (net.GetConstraint (current, i)
				.GetConstraint (indexes[current], indexes[i]) == false) {
				conflicts[current][i] = 0;
				if (conflicts[current][current] < i) {
					conflicts[current][current] = i;
				}
				return false;
			}
		}
		
		return true;
	}

	/*
	 *	Create the union of two sets which hold where the conflicts took place.
	 *	As described by Prosser, this is necessary to be able to continue
	 *	conflict-directed backjumping correctly to ensure the correct variable
	 *	will not be missed. The maximum in the set for variable "i" is retained
	 *	in position "conflicts[i][i]" for future reference. [CSPLib]
	 */

	private void UnionConflicts (int i, int j) {
		for (int n = 0; n < i; n++) {

			if ((conflicts[i][n] > -1) || (conflicts[j][n] > -1)) {
				conflicts [i][n] = 0;	//	true
			} else {
				conflicts [i][n] = -1;	//	false
			}

			if ((conflicts[i][n] > -1) && (conflicts[i][i] < n)) {
				conflicts[i][i] = n;
			}
		}
	}

	/*
	 *	Reset the sets between the given indexes to be empty. This allows
	 *	for new information to be stored in these sets without the old
	 *	information misleading the algorithm. [CSPLib]
	 */
	
	private void EmptyConflicts (int from, int to) {
		for (int i = from + 1; i <= to; i++)
			for (int j = 0; j <= i; j++)
				conflicts[i][j] = -1;
	}

	private void ClearSetup () {
		for (int i = 0; i < size; i++)
			for (int j = 0; j < size; j++)
				conflicts[i][j] = -1;	
	}
}
