package JCL;

/**
 * Implementation of a Backmarking & Conflict-Directed Backjumping solver for the 
 * Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last Update 14-11-96
 */

public class BM_CBJSolver extends Solver {

	int count = 0;
	/*
 	 * Data structures to hold the required information.  See the accompanying
 	 * research articles for complete descriptions of the structures. [CSLib]
 	 */

	int mcl[][];		//	N, K
	int mbl[];		//	N
	int conflicts[][];	//	N, N


	final static String NAME = "BackMarking & Conflict-Directed Backjumping hibrid";
	
	/*
	 *	Constructors.
	 */

	public BM_CBJSolver () {
	}

	public BM_CBJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {

		//	Initialitation of data structures.

		mcl = new int [size][MAX_DOMAIN];
		mbl = new int [size];
		conflicts = new int [size][size];
	
		ClearSetup (size, MAX_DOMAIN);

		Domain d = net.GetVariable (0).GetDomain ();
	
		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}

	/*
	 *	Solve the constraint network using the BACKMARKING with CONFLICT-DIRECTED
	 *	BACKJUMPING (BM-CBJ) method.  If this is the first variable then initialize
	 *	the data structures.  If a solution is found then update the "found"
	 *	variable, call the function to process the solution, and return the value
	 *	according to whether the first solution is desired or all solutions.  Then,
	 *	check if the timer has expired, and if it has, return immediately.
	 *	Otherwise, begin checking each possible instantiation of the variable.  For
	 *	each domain value, perform the following steps.  First, if preprocessing
	 *	eliminated the value, disregard the rest of the loop.  Otherwise,
	 *	instantiate the variable, check if the network is still consistent, and then
	 *	call the backtracking routine recursively.  After checking all possible
	 *	domain values, determine the variable to jump back to, update the minimum
	 *	backup level data structure, and return the variable to which the algorithm
	 *	should jump back. [CSPLib]
 	 */
	
	private int RecursiveSolve (int current, int k) {
		
		int i, h;	
	
		//	Local Variables
		int kk = 0;
		Variable v;

		int curr = count;

		/*
		try {
			Thread.sleep (500);	//	TEST
		} catch (InterruptedException e) {
		}
		*/

		NotifyEnterLevel (current);

		if (current >= size) {

			//	A solution has been found
			
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				count ++;
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		int jump;

		for (i = 0; i < k; i++) {
			if (net.GetConstraint (current, current).GetConstraint (i, i) == false) {
				System.err.println ("BM : variable not consistent with itself !");
				continue;
			}

			indexes[current] = i;

			NotifyInstanciation (current, i);

			if (IsConsistent (current)) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				jump = RecursiveSolve (current + 1, kk);
				if (jump != current) {
					NotifyLeaveLevel (current);
					return jump;
				}
			}
		}
		
		jump = (curr == count) ? conflicts[current][current] : current - 1;
		mbl[current] = jump;
		UnionConflicts (jump, current);
		for (i = jump + 1; i < size; i ++) {
			if (mbl[i] > jump)
				mbl[i] = jump;
		}
		EmptyConflicts (jump, current);
		NotifyLeaveLevel (current);
		return jump;
	}

	/*
	 *	Check if the current instantiation of the variables is consistent by
	 *	looking at the edge between the current variable and all previous variables.
	 *	However, this is not done if the algorithm has not backed up enough to
	 *	prevent a previous conflict from occurring again.  If a check is done
	 *	between two variables, this information is retained for later use. On the
	 *	other hand if a conflict exists, the set of conflicts with the current
	 *	variable is updated to reflect this fact. [CSPLib]
	 */

	private boolean IsConsistent (int current) {
		int i;

		if (mcl[current][indexes[current]] < mbl[current]) {
			conflicts[current][mcl[current][indexes[current]]] = 0;
			if (conflicts[current][current] < mcl[current][indexes[current]])
				conflicts[current][current] = mcl[current][indexes[current]];
			return false;
		}
		for (i = mbl[current]; i < current; i++) {
			mcl[current][indexes[current]] = i;
			
			NotifyConsistencyCheck ();

			if (net.GetConstraint (current, i)
				.GetConstraint (indexes[current], indexes[i]) == false) {
				
				conflicts[current][i] = 0;
				if (conflicts[current][current] < i)
					conflicts[current][current] = i;
				return false;
			}
		}
		return true;
	}


	/*
	 *	Create the union of two sets which hold where the conflicts took place.
	 *	As described by Prosser, this is necessary to be able to continue
	 *	conflict-directed backjumping correctly to ensure the correct variable
	 *	will not be missed. The maximum in the set for variable "i" is retained
	 *	in position "conflicts[i][i]" for future reference. [CSPLib]
	 */

	private void UnionConflicts (int i, int j) {
                for (int n = 0; n < i; n++) {
                        conflicts[i][n]
                                = ((conflicts[i][n] > -1) || (conflicts[j][n] > -1)) ? 0 : -1;

                        if ((conflicts[i][n] > -1) && (conflicts[i][i] < n)) 
                                conflicts[i][i] = n;
                }
        }

	/*
	 *	Reset the sets between the given indexes to be empty. This allows
	 *	for new information to be stored in these sets without the old
	 *	information misleading the algorithm. [CSPLib]
	 */
	
	private void EmptyConflicts (int from, int to) {
		for (int i = from + 1; i <= to; i++)
			for (int j = 0; j <= i; j++)
				conflicts[i][j] = -1;
	}
	/*
 	 * Clear the setup data structures (however, set them to the appropriate
	 * values).
 	 */

	private void ClearSetup (int n, int k) {
    		int i, j;

    		for (i = 0; i < n; i++) {
       	 		mbl[i] = 0;
			for (j = 0; j < k; j++)
        	    		mcl[i][j] = 0;
			for (j = 0; j < n; j++)
				conflicts[i][j] = -1;
    		}
	}

}
