package JCL;

/**
 * Implementation of a BackMarking and BackJumping hibrid solver for the 
 * Java Constraint Library.
 *
 * @author Marc Torrens
 * @version Last update 14-11-96
 */

public class BM_BJSolver extends Solver {
	
	boolean found = false;
	int count = 0;

	/*
 	 * Data structures to hold the required information.  See the accompanying
 	 * research articles for complete descriptions of the structures. [CSLib]
 	 */

	int mcl[][];
	int mbl[];
	int jump_place[];


	final static String NAME = "BackMarking & BackJumping hibrid";
	
	/*
	 *	Constructors.
	 */

	public BM_BJSolver () {
	}

	public BM_BJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {

		//	Initialitation of data structures.

		mcl = new int [size][MAX_DOMAIN];
		mbl = new int [size];
		jump_place = new int [size];
	
		ClearSetup (size, MAX_DOMAIN);

		Domain d = net.GetVariable (0).GetDomain ();

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}

 
	/*
	 *	Solve the constraint network using the BACKMARKING with BACKJUMPING (BM-BJ)
	 *	method.  If this is the first variable then initialize the data structures.
	 *	If a solution is found then update the "found" variable, call the function
	 *	to process the solution, and return the value according to whether the first
	 *	solution is desired or all solutions.  Then, check if the timer has expired,
	 *	and if it has, return immediately.  Otherwise, begin checking each possible
	 *	instantiation of the variable.  For each domain value, perform the following
	 *	steps.  First, if preprocessing eliminated the value, disregard the rest of
	 *	the loop.  Otherwise, instantiate the variable, check if the network is
	 *	still consistent, and then call the backtracking routine recursively.  After
	 *	checking all possible domain values, update the minimum backup level data
	 *	structure and return the variable to which the algorithm should jump back.
	 *	[CSPLib]
	 */
	
	private int RecursiveSolve (int current, int k) {
		
		int i, h;	
	
		//	Local Variables
		int kk = 0;
		Variable v;
		/*
		try {
			Thread.sleep (500);	//	TEST
		} catch (InterruptedException e) {
		}
		*/

		NotifyEnterLevel (current);

		if (current < size) {
			jump_place[current] = -1;
		} else {	//	current >= size

			//	A solution has benn found
		
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		for (i = 0; i < k; i++) {
			if (net.GetConstraint (current, current).GetConstraint (i, i) == false) {
				System.err.println ("BM : variable not consistent with itself !");
				continue;
			}

			indexes[current] = i;

			NotifyInstanciation (current, i);

			if (IsConsistent (current)) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				int jump = RecursiveSolve (current + 1, kk);
				if (jump != current) {
					NotifyLeaveLevel (current);
					return jump;
				}
			}
		}
		
		h = jump_place [current];
		mbl[current] = h;
		for (i = h + 1; i < size; i++)
			mbl[i] = Math.min (mbl[i], h);

		NotifyLeaveLevel (current);
		return jump_place [current];
	}

	/*
	 *	Check if the current instantiation of the variables is consistent by
	 *	looking at the edge between the current variable and all previous variables.
	 *	However, this is not done if the algorithm has not backed up enough to
	 *	prevent a previous conflict from occurring again.  If a check is done
	 *	between two variables, this information is retained for later use. On the
	 *	other hand if a conflict exists, the set of conflicts with the current
	 *	variable is updated to reflect this fact. [CSPLib]
	 */

	private boolean IsConsistent (int current) {
		int i;

		if (mcl[current][indexes[current]] < mbl[current])
			return false;
		for (i = mbl[current]; i < current; i++) {
			mcl[current][indexes[current]] = i;
			
			NotifyConsistencyCheck ();

			if (net.GetConstraint (current, i)
				.GetConstraint (indexes[current], indexes[i]) == false) {

				if (i > jump_place[current])
					jump_place[current] = i;
		
				return false;
			}
		}
		jump_place[current] = current -1;
		return true;
	}

	/*
 	 * Clear the setup data structures (however, set them to the appropriate
	 * values).
 	 */

	private void ClearSetup (int n, int k) {
    		int i, j;

    		for (i = 0; i < n; i++) {
			jump_place [i] = -1;  // !!!!!!! was 0 in CSPLib
       	 		mbl[i] = 0;
			for (j = 0; j < k; j++)
        	    		mcl[i][j] = 0;
    		}
	}

}
