package JCL;

/**
 * Implementation of a BackJumping solver for the Java
 * Constraint Library.
 *
 * @author Erik Bruchez
 * @version Last Update 14-11-96
 */

public class BJSolver extends Solver {
	
	int jump_place[];

	final static String NAME = "Simple Backjumping";
	
	/*
	 *	Constructors.
	 */

	public BJSolver () {
	}

	public BJSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		
		//	Initialize variables
		
		jump_place = new int[size];

		ClearSetup ();
		
		//	Start recursion
		Domain d = net.GetVariable (0).GetDomain ();

		try {
			RecursiveSolve (0, d.GetSize ());
		} catch (SolutionFoundException e) {
		}
	}
	
	/*
	 *	Solve the constraint network using the BACKJUMPING (BJ) method. If a solution
	 *	is found then notify it and return the value according to whether the first
	 *	solution is desired or all solutions. Otherwise, begin checking each possible
	 *	instantiation of the variable. For each domain value, perform the following
	 *	steps. First, if preprocessing eliminated the value, disregard the rest of
	 *	the loop. Otherwise, instantiate the variable, check if the network is
	 *	still consistent, and then call the backtracking routine recursively. After
	 *	checking all possible domain values, use jump_place to return the variable
	 *	to which the algorithm should jump back. [CSPLib]
	 */
	
	private int RecursiveSolve (int current, int k) {

		//	Local Variables
		int kk = 0;
		Variable v;

		NotifyEnterLevel (current);

		if (current == 0) {
			ClearSetup ();
		}
	
		if (current < size) {
			jump_place[current] = -1;
		} else {	//	current >= size

			//	A solution has been found
			
			NotifySolution ();

			if (FindMoreSolutions ()) {
				NotifyLeaveLevel (current);
				return size - 1;
			} else
				throw new SolutionFoundException ();
		}

		for (int i = 0; i < k; i++) {
			if (net.GetConstraint (current, current).GetConstraint (i, i) == false) {
				System.err.println ("BJ : variable not consistent with itself !");
				continue;
			}

			indexes[current] = i;
			NotifyInstanciation (current, i);

			if (IsConsistent (current)) {
				if (current < (size - 1)) {
					v = net.GetVariable (current + 1);
					kk = v.GetDomain ().GetSize ();
				}
				int jump = RecursiveSolve (current + 1, kk);
				if (jump != current) {
				//	NotifyLeaveLevel (current);
/*					for (int j = current; j >= jump; j--) {
						NotifyLeaveLevel (j);
					}
*/			
					return jump;
				}
			}
		}

		//NotifyLeaveLevel (current);
		for (int j = current; j > jump_place[current]; j--) {
			NotifyLeaveLevelAux (j);
		}
		NotifyBackjump();
		
		return jump_place[current];
	}

	/*
	 *	Check if the current instantiation of the variables is consistent and
	 *	update the jump back place array appropriately. If a conflict is found
	 *	and it is closer to the current variable, store that value. Otherwise,
	 *	if no conflict is found, the "conflict value" is the previous variable,
	 *	by default. [CSPLib]
	 */

	private boolean IsConsistent (int current) {
		for (int i = 0; i < current; i++) {
			NotifyConsistencyCheck ();

			if (net.GetConstraint (current, i)
				.GetConstraint (indexes[current], indexes[i]) == false) {
				if (i > jump_place[current])
					jump_place[current] = i;
				return false;
			}
		}

		jump_place[current] = current - 1;	//	TO VERIFY : can be -1 ?
		return true;
	}

	private void ClearSetup () {
		for (int i = 0 ; i < size; i ++ ) 
			jump_place [i] = -1;
	}  
}
