package JCL;

/**
 * Implementation of a simple Arc Consistency achieving solver 
 * for the Java Constraint Library.
 *
 * @author Marc Torrens
 * @version 18-11-96
 */

public class ACSolver extends SolverConsistency {
	
	boolean found = false;
	int count = 0;

	final static String NAME = "Arc Consistency";
	
	int stack[];
	boolean on_stack[];
	int top;
	
	/*
	 *	Constructors.
	 */

	public ACSolver () {
	}

	public ACSolver (Network net) {
		SetNetwork (net);
	}

	/**
	 *	Return the name of the algorithm.
	 */

	public String GetName () {
		return new String (NAME);
	}
	
	/**
	 *	Try to solve the problem.
	 */

	public void Solve () {
		try {
			ArcConsistency ();
		} catch (SolutionFoundException e) {
		}
	}

	/*
	 *	Perform arc consistency on the network.
	 */

	private void ArcConsistency () {

		//	Perform arc consistency
	
		ac = Achieve ();

		super.SetNetwork (this.net);
	}

	/*
	 *	Function to perform complete arc consistency on entire constraint network.
	 *	The process considers each "half" of the constraint network separate as
	 *	also done by Mackworth. It processes each edge using a stack to store
	 *	edges still to be examined. It first pushes all "edges" to be examined.
	 *	Then, it processes the edges one by one, placing variables back on the
	 *	stack if necessary (i.e. their domain values have changed). Note that
	 *	edges simply correspond to the edges leading into the variable stored on
	 *	the stack (thus only these variables need to be stored instead of all
	 *	edges). The function returns whether the network is consistent or not by
	 *	checking if any variables have their domains empty. Note that if a variable
	 *	x has any of its domain values eliminated, this is noted in the constraint
	 *	C[x][x], by eliminating this value for the original "I" matrix. [CSPLib]
	 */

	private boolean Achieve () {

		ClearStack ();

		for (int i = 0; i < size; i++)
			Push (i);
		
		while (!StackEmpty ()) {
			int j = Pop ();
			
			for (int i = 0; i < size; i++) {
				NotifyRevise (i, j);
				if ( (i != j) && (Revise (i, j) != 0))
					Push (i);
			}
		}

		for (int i = 0; i < size; i++) {
			int count = 0;

			Variable v = net.GetVariable (i);
			Domain d = v.GetDomain ();

			int i_domain_size = d.GetSize ();

			for (int j = 0; j < i_domain_size; j++)
				if (net.GetConstraint (i, i).GetConstraint (j, j))
					count++;

			if (count == 0) {
				System.out.println ("Warning ! The variable " + i + " has no values !");
				return false;
			}
		}

		return true;
	}
	
	/*
	 *	Clear and create the stack.
	 */

	private void ClearStack () {
		if (stack == null)
			stack = new int[size];
		if (on_stack == null)
			on_stack = new boolean[size];

		top = -1;

		for (int i = 0; i < size; i++)
			on_stack[i] = false;
	}

	/*
	 *	Push an integer on the stack.
	 */

	private void Push (int a) {
		if (!on_stack[a]) {
			top++;
			stack[top] = a;
			on_stack[a] = true;
		}
	}

	/*
	 *	Pop an integer from the stack.
	 */

	private int Pop () {
		int a = stack[top];
		on_stack[a] = false;
		top--;

		return a;
	}

	/*
	 *	Return true if the stack is empty.
	 */

	private boolean StackEmpty () {
		return top == -1;
	}

	/*
	 *	This function determines whether given a specific edge, there exists
	 *	some instantiation for "y" given the instantiation for "x". It loops
	 *	through all the value of the "y" variable. [CSPLib]
	 */

	private boolean Value (int x, int y, int i) {
		Variable v = net.GetVariable (y);
		Domain d = v.GetDomain ();

		int y_domain_size = d.GetSize ();

		for (int j = 0; j < y_domain_size; j++) {
			if ( (net.GetConstraint (x, y).GetConstraint (i, j))
			  && (net.GetConstraint (y, y).GetConstraint (j, j)) )
				return true;
		}

		return false;
	}


	/*
	 *	Determine which domain values for variable "x" can be eliminated by
	 *	considering the edge between "x" and "y". The number of values
	 *	deleted from the domain is returned. Each possible domain value of
	 *	"x" is considered separately in the loop. [CSPLib]
	 */

	private int Revise (int x, int y) {
		int del = 0;

		Variable v = net.GetVariable (x);
		Domain d = v.GetDomain ();

		int x_domain_size = d.GetSize ();

		for (int i = 0; i < x_domain_size; i++) {
			Constraint c = net.GetConstraint (x, x);

			if (c.GetConstraint (i, i) == false) {
				continue;
			} else {
				NotifyValue (x, y, i);
				if (Value (x, y, i) == false) {
					NotifyRemoveValue (x, i);
					NotifyRemoveValue ();
					RemoveValue (x, i);
					x_domain_size = net.GetVariable (x).GetDomain().GetSize();
					del++;
				}
			}
		}
		
		return del;
	}

	/*
	 *	Remove the valor i from the domain of the variable x
	 */

	private void RemoveValue (int x, int i) {
		Domain d;
		Domain d2;
		Variable v;
		Constraint c1,c2, c3;
		int dom1;
		int dom2;
		int l,m;	
	
		v = net.GetVariable (x);
		d = v.GetDomain ();
		String d2name = new String (d.GetName () + v.GetName ());
		int d2size = d.GetSize () - 1;		//	d2size could be 0
		d2 = new Domain (d2name, d2size);

		int count = 0;
		for (int k = 0; k < d.GetSize (); k ++) {
			if (k != i) {
				d2.SetValueName (count, d.GetValueName (k));
				count++;
			} 
		}

		//	d2 is the new domain for the variable x

		v.SetDomain (d2);
		
		//	Setup the constraints x, y

		dom1 = net.GetVariable (x).GetDomain ().GetSize ();
		for (int y = 0; y < size; y ++) {
			
			//	Constrain for the test
			c1 = net.GetConstraint (x, y);
			//	Constraint y, x
			c3 = new Constraint (net.GetVariable(y),net.GetVariable(x));
			//	Constraint x, y
			c2 = new Constraint (net.GetVariable(x),net.GetVariable(y));

			for (m = 0; m < dom1; m ++) {
				dom2 = net.GetVariable (y).GetDomain ().GetSize ();
				for (l = 0; l <  dom2; l ++) {
					if ((m == l) && (y == x)) {
						c2.SetConstraint (m, m);
						c3.SetConstraint (m, m);
					} else {
						if (m >= i) {
							//	Rotate the constraints
							if (c1.GetConstraint (m + 1, l)) {
								c2.SetConstraint (m, l);
								c3.SetConstraint (l, m);
							} else {
								//c1.ClearConstraint (m, l);
							}
						} else {
							//	Copy the constraints
							if (c1.GetConstraint (m , l)) {
								c2.SetConstraint (m, l);
								c3.SetConstraint (l, m);
							} else {
								//c1.ClearConstraint (m, l);
							}
						}
					}
				}
			}

			net.SetConstraint (c2);
			net.SetConstraint (c3);
		}
	}
	
	/*
	 *	Check if all the domains are empty
	 */
	
	private boolean AllDomainsEmpty () {
		boolean b = false;
		boolean dom_empty = false;
		
		for (int x = 0; x < size; x ++) {
			dom_empty = (net.GetVariable (x).GetDomain().GetSize() == 0);
			b = b || (!dom_empty);
		}
		
		return b;
	}


}
