package scalashop

import scalashop.common.measure

import java.awt.*
import java.awt.event.*
import javax.swing.*
import javax.swing.event.*

object ScalaShop:

  private val latch = java.util.concurrent.CountDownLatch(1)

  class ScalaShopFrame extends JFrame("ScalaShop\u2122"):
    setSize(1024, 600)
    setLayout(BorderLayout())

    val rightpanel = JPanel()
    rightpanel.setBorder(
      BorderFactory.createEtchedBorder(border.EtchedBorder.LOWERED)
    )
    rightpanel.setLayout(BorderLayout())
    add(rightpanel, BorderLayout.EAST)

    val controls = JPanel()
    controls.setLayout(GridLayout(0, 2, 0, 10))
    rightpanel.add(controls, BorderLayout.NORTH)

    val defaultsLabel = JLabel("Load Configuration")
    controls.add(defaultsLabel)

    val defaultsCombo = JComboBox(
      Array(
        "GameOfLife: Glider",
        "GameOfLife: Gosper Glider Gun",
        "GameOfLife: Pulsar"
      )
    )
    controls.add(defaultsCombo)

    // empty panel to get good spacing
    controls.add(new Panel())

    val defaultsLoader = JButton("Load Configuration")
    defaultsLoader.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          defaultsCombo.getSelectedItem.asInstanceOf[String] match
            case "GameOfLife: Gosper Glider Gun" =>
              canvas.loadFile(
                "src/main/resources/scalashop/gameoflife/gilder-gun.bmp"
              )
              filterCombo.setSelectedItem("game-of-life-step")
              scalingSpinner.setValue(20)
              autoStepSpinner.setValue(100)
            case "GameOfLife: Pulsar" =>
              canvas.loadFile(
                "src/main/resources/scalashop/gameoflife/pulsar.bmp"
              )
              filterCombo.setSelectedItem("game-of-life-step")
              scalingSpinner.setValue(30)
              autoStepSpinner.setValue(200)
            case "GameOfLife: Glider" =>
              canvas.loadFile(
                "src/main/resources/scalashop/gameoflife/glider.bmp"
              )
              filterCombo.setSelectedItem("game-of-life-step")
              scalingSpinner.setValue(40)
              autoStepSpinner.setValue(100)
    )
    controls.add(defaultsLoader)

    val filterLabel = JLabel("Filter")
    controls.add(filterLabel)

    val filterCombo = JComboBox(
      Array(
        "game-of-life-step"
      )
    )
    controls.add(filterCombo)

    val scalingLabel = JLabel("Integer Scaling")
    controls.add(scalingLabel)

    val scalingSpinner = JSpinner(SpinnerNumberModel(1, 1, 1000, 1))
    scalingSpinner.addChangeListener(
      new ChangeListener:
        def stateChanged(e: ChangeEvent): Unit =
          canvas.setScaling(scalingSpinner.getValue.asInstanceOf[Int])
    )
    controls.add(scalingSpinner)

    // cellular automaton controls
    val incrementLabel = JLabel("Auto-step Time (minimum, ms)")
    controls.add(incrementLabel)

    val autoStepSpinner = JSpinner(SpinnerNumberModel(200, 30, 2000, 50))
    controls.add(autoStepSpinner)

    val stepbutton = JButton("Apply filter")
    stepbutton.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          val time = measure {
            canvas.applyFilter(getFilterName)
          }
          updateInformationBox(s"Time: $time ms")
    )
    controls.add(stepbutton)

    val clearButton = JButton("Reload")
    clearButton.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          canvas.reload()
    )
    controls.add(clearButton)

    val autoStepButton = JButton("Auto-step")
    autoStepButton.addActionListener(
      new ActionListener:
        private var autoStepping: Boolean = false
        private val autoSteppingTimer = Timer(
          autoStepSpinner.getValue.asInstanceOf[Int],
          new ActionListener:
            def actionPerformed(e: ActionEvent): Unit =
              val time = measure {
                canvas.applyFilter(getFilterName)
              }
              updateInformationBox(s"Frame time: $time ms")
        )
        private def setControls(value: Boolean): Unit =
          filterCombo.setEnabled(value)
          stepbutton.setEnabled(value)
          clearButton.setEnabled(value)
          scalingSpinner.setEnabled(value)
          autoStepSpinner.setEnabled(value)
          defaultsCombo.setEnabled(value)
          defaultsLoader.setEnabled(value)

        private def enableControls(): Unit = setControls(true)
        private def disableControls(): Unit = setControls(false)

        def actionPerformed(e: ActionEvent): Unit =
          if autoStepping then
            autoSteppingTimer.stop()
            autoStepping = false
            enableControls()
            autoStepButton.setText("Auto-step")
          else
            autoStepping = true
            disableControls()
            autoStepButton.setText("Stop auto-step")
            autoSteppingTimer.setDelay(
              autoStepSpinner.getValue.asInstanceOf[Int]
            )
            autoSteppingTimer.start()
    )
    controls.add(autoStepButton)

    val info = JTextArea("   ")
    info.setBorder(BorderFactory.createLoweredBevelBorder)
    rightpanel.add(info, BorderLayout.SOUTH)

    val mainMenuBar = JMenuBar()

    val fileMenu = JMenu("File")
    val openMenuItem = JMenuItem("Open...")
    openMenuItem.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          val fc = JFileChooser()
          if fc.showOpenDialog(
              ScalaShopFrame.this
            ) == JFileChooser.APPROVE_OPTION
          then canvas.loadFile(fc.getSelectedFile.getPath)
    )
    fileMenu.add(openMenuItem)
    val exitMenuItem = JMenuItem("Exit")
    exitMenuItem.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          latch.countDown()
          dispose()
    )
    fileMenu.add(exitMenuItem)

    mainMenuBar.add(fileMenu)

    val helpMenu = JMenu("Help")
    val aboutMenuItem = JMenuItem("About")
    aboutMenuItem.addActionListener(
      new ActionListener:
        def actionPerformed(e: ActionEvent): Unit =
          JOptionPane.showMessageDialog(
            null,
            "ScalaShop, the ultimate image manipulation tool\nBrought to you by EPFL, 2015-2023"
          )
    )
    helpMenu.add(aboutMenuItem)

    mainMenuBar.add(helpMenu)

    setJMenuBar(mainMenuBar)

    val canvas = PhotoCanvas()

    val scrollPane = JScrollPane(canvas)

    add(scrollPane, BorderLayout.CENTER)
    setVisible(true)

    def updateInformationBox(time: String): Unit =
      info.setText(time)

    def getFilterName: String =
      filterCombo.getSelectedItem.asInstanceOf[String]

    // handle closing gracefully with a signal
    addWindowListener(
      new WindowAdapter:
        override def windowClosing(e: WindowEvent): Unit =
          latch.countDown()
          dispose()
    )

  try UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName())
  catch
    case _: Exception =>
      println("Cannot set look and feel, using the default one.")

  val frame = ScalaShopFrame()

  def main(args: Array[String]): Unit =
    frame.repaint()

    frame.defaultsLoader.doClick()

    // wait for a signal and block the main thread
    try latch.await()
    catch
      case _: InterruptedException =>
        println("Interrupted while waiting for the window to close.")
