import numpy as np
import PIL.Image  # type: ignore

# Type definitions
Image = np.ndarray
Kernel = np.ndarray
Position = tuple[int, int]
Size = tuple[int, int]

# Constant definitions
Infinity = float("inf")

np.set_printoptions(threshold=200)

MINIPROJECT_OPTIONS = {
    "skip_output_images": False,
}


def clamp(minimum: int, maximum: int, value: int) -> int:
    """Clamps a value between the theses bounds."""
    return min(max(value, minimum), maximum)


def split_name_ext(filename: str) -> tuple[str, str]:
    """Split a filename (or filepath) into its name or path without the extension, and its and extension separately"""
    dot_position = filename.rfind(".")
    if dot_position == -1:
        return filename, ""
    return filename[:dot_position], filename[dot_position:]


def load_image(filename: str) -> Image:
    """Load an image from a file and returns it as a numpy array"""
    print(">> Reading image from", filename)
    return np.array(PIL.Image.open(filename))


def save_image(img: Image, filename: str) -> None:
    """Save an image to a file"""
    if MINIPROJECT_OPTIONS["skip_output_images"]:
        print("** Skipping output image", filename)
        return
    print("<< Writing image to", filename)
    if not is_rgb(img):
        # convert to 0-255 range
        img = np.uint8(img)  # type: ignore
    PIL.Image.fromarray(img).save(filename)


def dimensions(img: Image) -> tuple[int, int]:
    """Return the dimensions of an image as a tuple (height, width)"""
    return img.shape[0], img.shape[1]


def new_image_gray(height: int, width: int) -> Image:
    """Create a new grayscale image with the given dimensions"""
    # could be uint8, but we use int16 to allow negative values
    return np.zeros((height, width), dtype=np.int16)


def new_image_gray_with_data(data: list[list[int]]) -> Image:
    """Create a new grayscale image with the given pixel values"""
    # could be uint8, but we use int16 to allow negative values
    return np.array(data, dtype=np.int16)


def new_random_gray_image(height: int, width: int) -> Image:
    """Create a new grayscale image with random pixel values"""
    return np.random.randint(0, 256, (height, width), dtype=np.uint16)


def new_random_rgb_image(height: int, width: int) -> Image:
    """Create a new RGB image with random pixel values"""
    return np.random.randint(0, 256, (height, width, 3), dtype=np.uint8)


def new_image_rgb(height: int, width: int) -> Image:
    """Create a new RGB image with the given dimensions"""
    return np.zeros((height, width, 3), dtype=np.uint8)


def is_rgb(img: Image) -> bool:
    """Return True if the image is RGB, False if it is grayscale"""
    return len(img.shape) == 3


def copy_image(img: Image) -> Image:
    """Return a copy of the given image"""
    return np.copy(img)
