#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include "util.h"
#include "const.h"
#include "engine.h"

FILE *logging = NULL;

#define REC_SEPARATOR "\n"

int log_init(char *file_name)
{
    logging = fopen(file_name, "w");
    return logging != NULL;
}

void log_info(const char *message)
{
    if (logging)
    {
        char buff[200];
        time_t ltime;       /* calendar time */
        ltime = time(NULL); /* get current cal time */
        struct tm *tm_info;
        tm_info = localtime(&ltime);

        strftime(buff, sizeof buff, "%Y-%m-%d %H:%M:%S", tm_info);
        fprintf(logging, "[%s] %s\n", buff, message);
    }
}

void log_stop()
{
    if (logging)
    {
        fclose(logging);
        logging = NULL;
    }
}

int **allocate_matrix(int r, int c)
{
    int **mat = calloc(r, sizeof(int *));
    for (int i = 0; i < r; i++)
    {
        mat[i] = calloc(c, sizeof(int));
    }
    return mat;
}

void destroy_matrix(int **mat, int r)
{
    for (int i = 0; i < r; i++)
    {
        free(mat[i]);
    }
    free(mat);
}

void clone_piece(const struct piece *piece, struct piece *new_piece)
{
    new_piece->size = piece->size;
    new_piece->shape = allocate_matrix(piece->size, piece->size);
    for (int i = 0; i < new_piece->size; i++)
    {
        for (int j = 0; j < new_piece->size; j++)
        {
            new_piece->shape[i][j] = piece->shape[i][j];
        }
    }
}

struct piece **load_pieces(const char *filename, int *num_pieces)
{
    FILE *f = fopen(filename, "r");

    if (!f)
    {
        return NULL;
    }

    fscanf(f, "%d", num_pieces);
    struct piece **pieces = malloc(*num_pieces * sizeof(struct piece *));

    for (int i = 0; i < *num_pieces; i++)
    {
        pieces[i] = malloc(4 * sizeof(struct piece));
        fscanf(f, "%d", &pieces[i][0].size);
        pieces[i][0].shape = allocate_matrix(pieces[i][0].size, pieces[i][0].size);

        for (int j = 0; j < pieces[i][0].size; j++)
        {
            for (int k = 0; k < pieces[i][0].size; k++)
            {
                fscanf(f, "%d", &pieces[i][0].shape[j][k]);
                pieces[i][0].shape[j][k] *= i + 1;
            }
        }

        for (int rot = 1; rot < 4; rot++)
        {
            clone_piece(&pieces[i][rot - 1], &pieces[i][rot]);
            rotate_piece(&pieces[i][rot - 1], &pieces[i][rot]);
        }
    }

    fclose(f);
    return pieces;
}

void merge_piece(
    struct board *board,
    const struct piece *piece,
    int piece_r,
    int piece_c)
{
    for (int i = 0; i < piece->size; i++)
    {
        for (int j = 0; j < piece->size; j++)
        {
            if (piece->shape[i][j])
            {
                board->data[piece_r + i][piece_c + j] = piece->shape[i][j];
            }
        }
    }
}

int init_state(
    struct state *pstate,
    int argc,
    char **argv)
{
    srand((unsigned int)time(NULL));

    pstate->board.n_rows = DEFAULT_ROWS;
    pstate->board.n_cols = DEFAULT_COLS;
    pstate->quit = 0;
    pstate->game_over = 0;
    pstate->pause = 0;
    pstate->debug = 0;
    pstate->score = 0;
    pstate->delay = DELAY;
    pstate->auto_delay = AUTO_DELAY;
    pstate->log = 0;
    pstate->autopilot = 0;

    pstate->pieces = load_pieces("pieces.txt", &pstate->num_pieces);
    pstate->current_piece = -1;
    pstate->piece_r = 0;
    pstate->piece_c = 0;
    pstate->piece_rot = 0;

    int c;
    while ((c = getopt(argc, argv, "DL:c:r:h")) != -1)
    {
        switch (c)
        {
        case 'L':
            if (!log_init(optarg))
            {
                printf("Cannot open log file %s\n", optarg);
                return 1;
            }
            else
            {
                pstate->log = 1;
            }
            break;
        case 'D':
            pstate->debug = 1;
            break;
        case 'c':
            pstate->board.n_cols = atoi(optarg);
            if (pstate->board.n_cols == 0)
            {
                pstate->board.n_cols = DEFAULT_COLS;
                printf("Cannot parse size\n");
            }
            printf("Size (cols) = %d\n", pstate->board.n_cols);
            break;
        case 'r':
            pstate->board.n_rows = atoi(optarg);
            if (pstate->board.n_rows == 0)
            {
                pstate->board.n_rows = DEFAULT_ROWS;
                printf("Cannot parse size\n");
            }
            printf("Size (rows) = %d\n", pstate->board.n_rows);
            break;
        case 'h':
        case '?':
            break;
        }
    }
    pstate->board.data = allocate_matrix(pstate->board.n_rows, pstate->board.n_cols);

    return 0;
}

void destroy_state(struct state *pstate)
{
    for (int i = 0; i < pstate->num_pieces; i++)
    {
        for (int rot = 0; rot < 4; rot++)
        {
            destroy_matrix(
                pstate->pieces[i][rot].shape,
                pstate->pieces[i][rot].size);
        }
        free(pstate->pieces[i]);
    }
    free(pstate->pieces);

    destroy_matrix(
        pstate->board.data,
        pstate->board.n_rows);

    if (pstate->log)
    {
        log_stop();
    }
}