#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>

#include "const.h"
#include "engine.h"
#include "util.h"
#include "display.h"

void info(struct state *pstate, const char *message)
{
    move(4, 0);
    clrtoeol();
    mvprintw(4, 0, message);

    if (pstate->log)
    {
        log_info(message);
    }

    if (pstate->debug)
    {
        refresh();
        wgetch(pstate->menu_win);
    }
}

int is_action_key(int c)
{
    return (c == KEY_UP ||
            c == KEY_DOWN ||
            c == KEY_LEFT ||
            c == KEY_RIGHT ||
            c == ' ');
}

void process_keys(
    struct state *pstate,
    int c)
{
    int arrow_key = is_action_key(c) && c != ' ';
    int new_piece_rot = pstate->piece_rot,
        new_piece_r = pstate->piece_r,
        new_piece_c = pstate->piece_c;

    switch (c)
    {
    case KEY_UP:
        info(pstate, "Up");
        new_piece_rot = (new_piece_rot + 1) % 4;
        break;

    case KEY_DOWN:
        info(pstate, "Down");
        new_piece_r = (new_piece_r + 1) % pstate->board.n_rows;
        break;

    case KEY_LEFT:
        info(pstate, "Left");
        new_piece_c--;
        break;

    case KEY_RIGHT:
        info(pstate, "Right");
        new_piece_c++;
        break;

    case ' ':
        info(pstate, "Space");
        if (pstate->current_piece >= 0 &&
            !pstate->pause)
        {
            pstate->piece_r = drop_piece(
                &pstate->board,
                &pstate->pieces[pstate->current_piece][pstate->piece_rot],
                pstate->piece_r,
                pstate->piece_c);
        }
        break;

    case 'D':
    case 'd':
        pstate->debug = 1 - pstate->debug;
        break;

    case '+':
        pstate->delay = max(pstate->delay - 100, 100);
        break;

    case '-':
        pstate->delay += 100;
        break;

    case '^':
        pstate->auto_delay = max(pstate->auto_delay - 100, 100);
        break;

    case '_':
        pstate->auto_delay += 100;
        break;

    case 'P':
    case 'p':
        pstate->pause = 1 - pstate->pause;
        break;

    case 27: // Escape key
    case 'q':
    case 'Q':
        pstate->quit = 1;
        break;

    case 'a':
    case 'A':
        pstate->autopilot = 1 - pstate->autopilot;
    }

    if (pstate->current_piece >= 0 &&
        !pstate->pause &&
        arrow_key &&
        is_valid_position(
            &pstate->board,
            &pstate->pieces[pstate->current_piece][new_piece_rot],
            new_piece_r,
            new_piece_c))
    {
        pstate->piece_rot = new_piece_rot;
        pstate->piece_r = new_piece_r;
        pstate->piece_c = new_piece_c;
    }
}

void process_autotick(struct state *pstate)
{
    static int moves = 0;
    static int steps[100];

    if (!pstate->autopilot || pstate->current_piece < 0 || pstate->game_over)
    {
        moves = 0;
        return;
    }

    if (moves > 0)
    {
        process_keys(pstate, steps[--moves]);
        if (moves == 0)
        {
            pstate->autopilot = -1; // done with the current piece, waiting for the new one
        }
    }

    if (moves == 0 && pstate->autopilot > 0 && pstate->current_piece >= 0)
    {
        int best_c = pstate->piece_c,
            best_rot = pstate->piece_rot;

        find_best_position(
            &pstate->board,
            pstate->pieces[pstate->current_piece],
            &best_c,
            &best_rot);

        int crt_r = pstate->piece_r,
            crt_c = pstate->piece_c,
            crt_rot = pstate->piece_rot;

        steps[moves++] = ' ';

        int delta_c = (best_c - crt_c >= 0) ? 1 : -1;
        while (crt_c != best_c)
        {
            steps[moves++] = (delta_c > 0 ? KEY_RIGHT : KEY_LEFT);
            crt_c += delta_c;
        }

        while (crt_rot != best_rot)
        {
            steps[moves++] = KEY_UP;
            crt_rot = (crt_rot + 1) % 4;
        }
    }
}

void process_tick(struct state *pstate)
{
    if (pstate->pause || pstate->game_over)
    {
        return;
    }

    if (pstate->current_piece < 0)
    {
        pstate->current_piece = rand() % pstate->num_pieces;

        pstate->piece_rot = 0;
        pstate->piece_r = 0;
        pstate->piece_c = pstate->board.n_cols / 2 -
                          pstate->pieces[pstate->current_piece][0].size / 2;

        if (pstate->autopilot)
        {
            pstate->autopilot = 1; // Make it positive
        }
    }

    if (!is_valid_position(
            &pstate->board,
            &pstate->pieces[pstate->current_piece][pstate->piece_rot],
            pstate->piece_r + 1,
            pstate->piece_c))
    {
        if (pstate->piece_r < 1)
        {
            pstate->game_over = 1;
        }
        else
        {
            merge_piece(
                &pstate->board,
                &pstate->pieces[pstate->current_piece][pstate->piece_rot],
                pstate->piece_r,
                pstate->piece_c);

            int lines = clear_lines(&pstate->board);
            pstate->score += 10 * lines;
            // Faster every 100 points -- disabled for now
            // pstate->delay = max(100, DELAY - pstate->score / 100 * 100);
        }
        pstate->current_piece = -1;
    }
    else
    {
        pstate->piece_r = (pstate->piece_r + 1) % pstate->board.n_rows;
    }
}

int main(int argc, char **argv)
{
    int c;
    struct state state;

    if (init_state(&state, argc, argv))
    {
        return 1;
    }
    init_display(&state);
    print_board(&state);

    int remaining_ms = state.delay;
    int auto_remaining_ms = AUTO_DELAY;
    while (!state.quit)
    {
        wtimeout(
            state.menu_win,
            max(10, min(remaining_ms, auto_remaining_ms)));
        struct timeval tval_before, tval_after, tval_result;

        gettimeofday(&tval_before, NULL);
        int c = wgetch(state.menu_win);
        gettimeofday(&tval_after, NULL);
        timersub(&tval_after, &tval_before, &tval_result);

        int time_passed = tval_result.tv_sec * 1000 + tval_result.tv_usec / 1000;

        remaining_ms -= time_passed;
        auto_remaining_ms -= time_passed;

        if (c == ERR)
        {
            if (remaining_ms < 0)
            {
                remaining_ms = state.delay;
                process_tick(&state);
            }

            if (auto_remaining_ms < 0)
            {
                auto_remaining_ms = state.auto_delay;
                process_autotick(&state);
            }
        }
        else
        {
            if (!state.autopilot || !is_action_key(c))
            {
                // Only process action keys outside of autopilot mode
                process_keys(&state, c);
            }
        }

        mvprintw(2, 0,
                 "Score: %6d %s",
                 state.score,
                 state.game_over ? "[GAME OVER]" : state.pause ? "[Paused]"
                                                               : "");
        move(3, 0);
        clrtoeol();
        print_board(&state);
        refresh();
    }
    endwin();
    destroy_state(&state);

    return 0;
}
