#include <iostream>
#include <vector>
using namespace std;


class Figure
{
public:
    Figure()
    {}

    virtual ~Figure()
    {}

    virtual void afficher(ostream& out) const = 0;

    // Pour faire de la copie polymorphique
    virtual Figure* copie()  const = 0;

    // ne pas permettre au monde extérieur
    // la copie par constructeur de copie
    // car un constructeur ne peut pas être
    // polymorphique
protected:
    Figure(const Figure& f) = default;
};

class Rectangle : public Figure
{
public:
    Rectangle(double l, double h)
        :largeur(l), hauteur(h)
    {}

    void afficher(ostream& out) const
    {
        out << "rectangle : " << largeur << "x" << hauteur <<endl;

    }
    Rectangle* copie() const override
    {
        return new Rectangle(*this);
    }
private:
    // Si ces attributs avaient été des pointeurs,
    // il aurait fallu penser à la copie profonde
    double largeur;
    double hauteur;
};


class Cercle : public Figure
{

public:
    Cercle(double r)
        :rayon(r)
    {}

    void afficher(ostream& out) const
    {
        out << "cercle de rayon " << rayon << endl;
    }
    Cercle* copie() const override
    {
        return new Cercle(*this);
    }
private:
    double rayon;
};


class Dessin
{

private:
    // pour le polymorphisme: on est obligé de travailler avec des
    // pointeurs sur Figure
    vector<Figure*> desFigures;

public:

    // 1ère version: le programmeur de la classe Dessin
    // se décharge du souci de la gestion des pointeurs
    // sur l'utilisateur de la classe (ici le main)
    /*
     void ajouterFigure(Figure* ptrF){
       desFigures.push_back(ptrF);
     }
     // mais attention, dans ce cas, il faut se préoccuper
     // de savoir si la classe Dessin a la propriété
     // (transfert de propriété depuis le code appelant)
     // dans ce cas la classe Dessin devrait a minima
     // vider le vecteur
    */
    /*
    // 2ème version: le programmeur de la classe Dessin
    // "cache" à l'utilisateur de sa classe qu'il lui faut des pointeurs
    // mais ici l'objet fourni au moment de l'appel peut ne pas être propre
    // à la collection (éventuellement partagé)
    // il n'est pas possible de mettre Figure comme type du paramètre
    // car dans cet exemple Figure est abstraite.
    // on peut mettre const Figure& mais dans ce cas le vecteur doit contenir
    // des références sur des objets constants : vector <const Figure*>
    // ce qui n'est pas souhaitable si on veut pouvoir changer les figures
    // du dessin par exemple pour les colorier.
    void ajouterFigure(Figure& f)
    	{
    		desFigures.push_back(&f);
    	}

    */

    // 3ème version: le programmeur de la classe Dessin "cache" à l'utilisateur
    // de la classe qu'il lui faut des pointeurs et assure que la figure
    // soit propre à la collection (pas un objet éventuellement partagé
    // avec d'autres dessins)
    void ajouterFigure(const Figure& f)
    {
        // mais attention la copie doit être polymorphique
        // et les constructeurs (notamment de copie)
        // ne sont pas polymorphiques, il faut donc
        // une méthode de copie

        desFigures.push_back(f.copie());
    }


    // si on fait des copies ou si la propriété est transférée
    // il faut restituer la mémoire (ici  avec la version 3
    // ou avec la version 1 s'il est décidé que la propriété est transférée)

    virtual ~Dessin()
    {
        for (auto* figure : desFigures) {
            delete figure;
        }

        desFigures.clear();

    }

    void afficher(ostream& out) const
    {
        for (auto* figure : desFigures) {
            figure->afficher(out);
        }
    }

};


int main()
{
    Dessin dessin;

    Cercle c1(3.5);
    Rectangle r1(1.2, 4.5);

    // si l'on montre les pointeurs (version 1)
    //dessin.ajouterFigure(&c1);
    //dessin.ajouterFigure(&r1);
    // ou
    // dessin.ajouterFigure(new Cercle(3.4)); // s'il y a transfert de propriété


    // si on ne montre pas la propriété
    dessin.ajouterFigure(c1);
    dessin.ajouterFigure(r1);

    dessin.afficher(cout);

    return 0;
}

// Dans le cas où Dessin a la propriété des figures et que ces dernières
// ne sont pas amenées à être partagées avec d'autres parties de programme
// alors les pointeurs intelligents peuvent avantageusement être utilisés
// et déchargent le programmeur du souci de gérer la mémoire
// -> voir collect-heterogene-uniqueptr.cc









