% Problem 1

close all
clear all

% Define parameters
N = 256; % number of carriers
L = 20; % length of the cyclic prefix
Noise_variance = 0.3; % noise variance after the DFT at the receiver
T = 1e-6; % [seconds]

% Load various files
load pn_sequence.mat
load data_symbols.mat

% Get the indices of the carriers where we transmit data symbols.
frequencies = [30, 80, 110, -20, -70, -100]*1e6;
index_carriers = mod(frequencies*T, N) + 1
% Or in a fancier way
% [~, index_carriers] = ismembertol(frequencies, fftshift([-N/2:1:N/2-1]/T), 1e-6)

no_useful_carriers = length(index_carriers);

% Generate the training sequence (same symbol [1+1j] on all carriers)
training_sequence = ones(1, N) + 1j*ones(1,N);

% See if we need to do some zero padding to fit the Tx data symbols
no_ofdm_blocks = ceil(length(data_symbols)/no_useful_carriers);
zero_padding = no_ofdm_blocks*no_useful_carriers - length(data_symbols)
data_padded = [data_symbols, zeros(1,zero_padding)];

% Create the matrix with the data symbols
B = reshape(data_padded, no_useful_carriers, no_ofdm_blocks);

% We need one extra OFDM block for the training sequence
A = zeros(N, no_ofdm_blocks + 1);
% Insert the training sequence
A(:,1) = training_sequence;
% Insert the data symbols
A(index_carriers,2:end) = B;

% Do the IFFT
a0 = ifft(A,N);
% Add the CP
a = [a0(N-L+1:N,:); a0];

% Serialize the Tx samples
tx_samples = a(:);

% Add the P/N sequence
tx_samples = [pn_sequence, tx_samples.'];

% The ISI channel (sample-level)
h = [3.2, 1.7, 1.3, 1, 0.8, 0.45, 0.3, 0.15, 0.05];

% Convolve with the channel
samples = conv(tx_samples, h);

% Generate the noise samples
sigma2 = Noise_variance/N; % Noise variance is increased since the DFT is not an unitary transformation
sigma = sqrt(sigma2);
% Create the sample-level noise vector
noise = (sigma/sqrt(2))*randn(size(samples)) + 1i*(sigma/sqrt(2))*randn(size(samples));

% Add the noise, create the Rx samples
rx_samples = samples + noise;

% Find the P/N sequence
R = xcorr(rx_samples, pn_sequence);
R = R(length(rx_samples):end-length(pn_sequence)+1);
[~, index_start_pn] = max(abs(R))

% Remove the P/N sequence
rx_samples = rx_samples(index_start_pn + length(pn_sequence):end);

% Remove eventual garbage at the end
no_rx_ofdm_blocks = floor(length(rx_samples)/(N+L))
rx_samples = rx_samples(1:end-mod(length(rx_samples),N+L));

% Prepare the samples for the OFDM receiver
rx_samples = reshape(rx_samples, N+L, no_rx_ofdm_blocks);

% Remove the CP
rx_samples = rx_samples(L+1:end,:);
% Do the FFT
rx_fft = fft(rx_samples,N);

% Estimate and plot the channel in frequency domain (lambdas)
lambdas = rx_fft(:,1).'./training_sequence;

% Compute the theoretical lambdas
th_lamdas = fft(h, N);

% Plot the lambdas and check if they agree
figure;
plot(abs(lambdas), '*-');
grid on; hold on;
plot(abs(th_lamdas), 'o--');
title('The lambdas');
xlabel('Frequency index');
ylabel('abs(lambdas)');
legend('Estimated (LS)', 'Theoretical');

% Correct for the lambdas
lambdas = lambdas(index_carriers);
eq_symbols = rx_fft(index_carriers,2:end)./lambdas(:);

% Serialize
eq_symbols = eq_symbols(:).';

% Remove eventual tails
eq_symbols = eq_symbols(1:length(data_symbols));

% Plot the received symbols
figure;
plot(eq_symbols,'*'); grid on;
title('Equalized symbols at Rx');
xlabel('Real'); ylabel('Imag');

% Take decisions and compute SER
dec_symbols = (2*(real(eq_symbols)>0) - 1) + 1j*(2*(imag(eq_symbols)>0) - 1);
SER = sum(dec_symbols ~= data_symbols)/length(data_symbols)
