function Y_corr = sol_correctOFDMSymbolRotation(Y, pilot_indices, pilot_symbols)
%CORRECTOFDMSYMBOLROTATION uses the pilot symbols to estimate and remove
% the rotation of the OFDM symbols (after the DFT) 
%Y_CORR = CORRECTOFDMSYMBOLROTATION(Y, PILOT_INDICES, PILOT_SYMBOLS) 
%   Removes the block-dependent phase rotation in OFDM symbols by using 
%   the pilot symbols.
%
%   Y: a matrix in which, except for the rotation, the columns are 
%      noisy OFDM blocks (after DFT). Each element of a column is expected
%      to be rotated by the same phase. 
%   PILOT_INDICES: indices of pilot symbols. 
%   PILOT_SYMBOLS: vector of pilot symbols.
%   Y_CORR: the matrix Y corrected for the rotations.


global ofdmc;
if isempty(ofdmc)
    ofdmConfig();
end


num_carriers = size(Y,1);
num_symbols  = size(Y,2);

if ~isvector(pilot_indices)
    error('ofdm:invalidPilots', ...
        'PILOT_INDICES must be a vector');
end

if any(pilot_indices < 1 | pilot_indices > num_carriers)
    error('ofdm:invalidPilots', ...
        'PILOT_INDICES out of range');
end


if ~isvector(pilot_symbols)
    error('ofdm:invalidPilots', ...
        'PILOT_SYMBOLS must be a vector');
end

if numel(pilot_symbols) ~= numel(pilot_indices)
    error('ofdm:invalidPilots', ...
        'PILOT_INDICES and PILOT_SYMBOLS should have the same size');
end


% Here we implement the division method, just for testing purposes. 
% The function will output the LS estimate, see later in the code.
rotations    = angle(Y(pilot_indices,:)  ...
    ./ repmat(pilot_symbols(:), 1, num_symbols));

% ATTENUATION should be ideally close to 1
attenuations = abs(Y(pilot_indices,:)) ./ ...
    abs(repmat(pilot_symbols(:), 1, num_symbols));

if (ofdmc.verbose)
    fprintf('\nRotations of Each OFDM Block\n');
    fprintf('block');
    fprintf('\ttheta(%d)',pilot_indices);
    fprintf('\n');
    fprintf('-----------------------------------------\n');
    fmt = ['%d' repmat('\t%10.3g',1,numel(pilot_indices)) '\n'];
    fprintf(fmt,[[1:num_symbols]', rotations']');
    
    fprintf('\n\n\nAttenuations of Each OFDM block\n');
    fprintf('symbol');
    fprintf('\tmag(%d)',pilot_indices);
    fprintf('\n');
    fprintf('-----------------------------------------\n');
    fmt = ['%d' repmat('\t%10.3g',1,numel(pilot_indices)) '\n'];
    fprintf(fmt,[[1:num_symbols]', attenuations']');
end


% Here we do the LS estimation of phase rotation based on multiple pilot
% carriers. The estimate of the phase is done in each OFDM block
% independently. (As opposed to the LS approximation that takes several
% estimates (of several blocks) into account and finds the best linear fit.

rotations  =  angle(pilot_symbols(:)' * Y(pilot_indices,:));

if (ofdmc.verbose)
    fprintf('\nRotation Correction Each OFDM Symbol\n');
    fprintf('symbol');
    fprintf('\ttheta');
    fprintf('\n');
    fprintf('-----------------------------------------\n');
    fmt = ['%d\t%10.3g\n'];
    fprintf(fmt,[[1:num_symbols]', rotations']');
end

phase_corr = repmat(rotations,num_carriers,1);

Y_corr = Y .* exp(-1j*phase_corr);


end

