function est_bits = receiver(rx_signal)
%EST_BITS = RECEIVER(RX_SIGNAL) estimates the sent bits from RX signal.
% 
%  The receiver proceeds through the following steps:
%  1. The first step is to find the start of data symbols and
%     carrier frequency offset (CFO), by using the synchronization 
%     sequence (PREAMBLE defined in OFDMC); and correct the CFO.
%     
%  2. Next, the time-domain OFDM symbols are converted to frequency
%     domain using OFDM_RX_FRAME function.
% 
%  3. Channel coefficients are estimated (in frequency domain) using
%     CHANNEL_EST LS estimator. Then, the symbols are equalized.
%
%  4. The residual rotation in OFDM symbols (due to CFO) is corrected 
%     with the help of pilot carriers.
% 
%  5. After removing the training sequence and the pilot carriers, data
%  bits are estimated using a minimum-distance decoder.

global ofdmc;
if isempty(ofdmc)
    ofdmConfig();
end

function_mapper_usrp; % initializes function handles

% 1. Find the starting time of data, and estimate the CFO

[tau_estim, cfo_estim] = estimateTauAndCFO(rx_signal, ofdmc.preamble, ...
    ofdmc.T, ofdmc.rangeCFO);

% 1.1. Correct for CFO
rx_signal = rx_signal.* ...
    exp(-1j*2*pi*cfo_estim*(0:numel(rx_signal)-1)*ofdmc.T);

% 1.2. Skip the synchro and zero sequence
tau_estim = tau_estim + numel(ofdmc.preamble) + ofdmc.nZeros;

% 1.3. Compute the OFDM data length (including the training OFDM symbols)
ofdmDataLength = ofdmc.nOFDMSymbols*(ofdmc.nCarriers + ofdmc.cpLength);

%     We might have several peaks in xcorr() because we repeat the
%     data. To be sure we do not get out of the Rx data, we back up
if tau_estim + ofdmDataLength - 1 > length(rx_signal)
   tau_estim =  tau_estim - ofdmc.txSignalLength;
   if (ofdmc.verbose)
    fprintf('We have backed up to the previous peak of xcorr()... \n');
   end
end

% 1.4. Extract OFDM symbols
rx_data = rx_signal(tau_estim:tau_estim + ofdmDataLength - 1);

if (ofdmc.verbose)
    tfplotReImPhase(rx_data, 1/ofdmc.T,'r','RX Signal (Data Part)');
end

% 1.5. Extract the noise and estimate the noise variance
noise_samples = rx_signal(tau_estim - ofdmc.nZeros:tau_estim - 1);
noiseVar      = var(noise_samples);
sigma2        = ofdmc.nCarriers*noiseVar; % We have N = ofdmc.noCarriers carriers.


% 2. Convert OFDM symbols to frequency domain
Y = sol_ofdm_rx_frame(rx_data, ofdmc.nCarriers, ofdmc.psdMask, ofdmc.cpLength);

% 3. Estimate the channel and equalize the symbols
% 3.1. Channel estimation
lambda = sol_channel_estLS(Y, ofdmc.trainingSymbols);

if (ofdmc.verbose)
    % plot the esimated channel response
    lambda_aug = zeros(1,ofdmc.nCarriers);
    lambda_aug(ofdmc.psdMask) = lambda;
    figure;
    carrierLine = -ofdmc.nCarriers/2:1:ofdmc.nCarriers/2-1;
    plot(carrierLine, abs(fftshift(lambda_aug)),'s-'); 
    grid on; 
    hold on;
    plot(carrierLine, angle(fftshift(lambda_aug)),'*-');
    legend('Magnitude','Phase');
    title('Estimated channel (LS)'); 
    xlabel('Carrier index'); 
    ylabel('Magnitude and Phase');
end

% 3.2. Equalization
Y_eq =  Y .* repmat(1./lambda(:), 1, ofdmc.nOFDMSymbols);

% 4. Correct the rotation due to the residual CFO
Y_aug = zeros(ofdmc.nCarriers, ofdmc.nOFDMSymbols);
Y_aug(ofdmc.psdMask,:) = Y_eq;

Y_aug = correctOFDMSymbolRotation(Y_aug, ofdmc.pilotIndices, ofdmc.pilotSymbols);

% 5. Estimate the data bits 
% 5.1. Remove the training sequence
Y_aug  = Y_aug(:,2:end);
% 5.2. Remove the pilot symbols and masked carriers
Y_data = Y_aug(ofdmc.dataMask, :);
% 5.3. Serialize the Rx data symbols
rx_data_symbols = Y_data(:);

if (ofdmc.verbose)
    scatterplot(rx_data_symbols);
    grid on;
    title('RX Constellation');
end
% Get the mapping
switch(lower(ofdmc.constellationType))
    case 'qam'
        mapping = sol_qamMap(ofdmc.M);
    case 'psk'
        mapping = sol_pskMap(ofdmc.M);
    otherwise
        error('ofdm:invalidModulation','Unsupported modulation type %s', ...
            ofdmc.modtype);
end
% Normalize constellation to unit average energy - not really necessary
mapping      = mapping./sqrt(mean(abs(mapping).^2));

est_symbols = sol_decoder(rx_data_symbols, mapping);
est_bits    = transpose(sol_de2bi(est_symbols)); % transpose to get it ready for serializing over rows (compatible with Python)
% N.B.: in Python, the output of de2bi is already serialized over rows
est_bits    = est_bits(:); % (:) is serializing over columns, which are now rows since we used transpose above

end

