function Rf = sol_ofdm_rx_frame(rx_symbols, num_carriers, psd_mask, prefix_length)
% MY_OFDM_RX_FRAME Receiver for ODFM signals (without channel equalization)
%  RF = MY_OFDM_RX_FRAME(RX_SYMBOLS, NUM_CARRIERS, PSD_MASK, PREFIX_LENGTH)
%
%       RX_SYMBOLS: vector of channel outputs. It is a filtered and noisy
%       version of the transmitted symbols. 
%       NUM_CARRIERS: number of carriers per OFDM block (power of 2). The
%       same as in OFDM_TX_FRAME.
%       PSD_MASK: A {0,1}-valued vector of length NUM_CARRIERS. The same 
%       as in OFDM_TX_FRAME.
%       PREFIX_LENGTH: cyclic-prefix length (in number of samples). The
%       same as in OFDM_TX_FRAME.
%
%   RF: Matrix of TRAINING_SYMBOLS and DATA_SYMBOLS (see OFDM_TX_FRAME) as
%   seen at the output of the equivalent "parallel channels" created by
%   OFDM. To obtain the RF matrix, we start by rearranging the received
%   symbols columnwise in a matrix that has (NUM_CARRIERS + PREFIX_LENGTH)
%   rows. In doing so, we remove the tail of RX_SYMBOLS that don't fill an
%   entire column. Then we remove the cyclic prefix and take the FFT.
%   Finally, the symbols that correspond to unused carriers are removed.


if ~isscalar(num_carriers) || num_carriers < 0  || mod(num_carriers,1) ~= 0
    error('ofdm_rx_frame:dimensionMismatch', ...
        'NUM_CARRIERS must be a positive scalar integer');
end

if ~isscalar(prefix_length) || prefix_length < 0  || mod(prefix_length,1) ~= 0
    error('ofdm_rx_frame:dimensionMismatch', ...
        'PREFIX_LENGTH must be a positive scalar integer');
end


if ~ isvector(psd_mask)|| numel(psd_mask) ~= num_carriers
    error('ofdm_rx_frame:dimensionMismatch',...
        'PSD_MASK must be a vector of length %d', ...
        num_carriers);
end

if any(psd_mask ~= 0 & psd_mask ~= 1)
    error('ofdm_rx_frame:invalidMask',...
        'PSD_MASK must be {0,1}-valued');
end

psd_mask = logical(psd_mask);

rx_symbols = rx_symbols(:);

% trim the received data such that we have an integer number of OFDM symbols
num_ofdm_symbols = floor((length(rx_symbols))/(num_carriers+prefix_length));
rx_symbols = rx_symbols(1:num_ofdm_symbols*(num_carriers+prefix_length));

% "remove" cyclic prefix
rx_withCP = reshape(rx_symbols, prefix_length+num_carriers, num_ofdm_symbols); 
rx_noCP = rx_withCP(prefix_length+(1:num_carriers), :); % remove rows corresponding to cyclic prefix

% go to the frequency domain
Rf = fft(rx_noCP, num_carriers);

% remove the off-carrier symbols
Rf = Rf(psd_mask, :);