import math
import numpy as np
import mpmath as mp
import warnings
from typing import Tuple

import gpsc as gpsc

from sys import path
path.append('../ephemerides/')
from utilGpsEphemerides import Ephemeris


def limitValidRange(t_in: float) -> float:
    """
    Correct ephemerides reference time to be in the range [-302400, 302400].

    This subfunction adds or subtracts 604800 to limit the given time
    to be in the interval [-302400, 302400] in accordance with the GPS
    standard (see IS-GPS-200D Table 20-IV, p.97)
    """

    HALF_RANGE = 302400

    if t_in < -HALF_RANGE:
        return t_in + 2 * HALF_RANGE
    elif t_in > HALF_RANGE:
        return t_in - 2 * HALF_RANGE
    else:
        return t_in


def rad2deg(r: float) -> float:
    """Helper function: radians to degree conversion"""

    if r is None:
        raise ValueError('ecef2wgs84:rad2deg:wrongSyntax', 'Incorrect number of arguments')

    if isinstance(r, complex):
        warnings.warn('ecef2wgs84:rad2deg:complexArgument: Imaginary parts of complex ANGLE argument ignored',
                      RuntimeWarning)
        r = r.real

    return r * 180 / math.pi


def ecef2wgs84(x: float, y: float, z: float) -> Tuple[float, float, float]:
    """
    Computes the user position in ellipsoid coordinates.

    Accepts the position of the user (receiver) in the ECEF coordinate system and uses the WGS-84 model
    to compute the corresponding ellipsoid coordinates.

    :return: (long, lat, h) - the longitude, latitude and height correspondingly
    """

    # WGS-84 constants
    EP = 0.00335281066474
    AE = 6378137

    # Tolerance for iteration
    epsilon = 1e-3

    # Compute distance from earth
    r = np.linalg.norm(np.array([x, y, z]))

    # Longitude
    long = rad2deg(math.atan(y / x))

    # Latitude
    lc = math.atan(z / np.linalg.norm(np.array([x, y])))    # Would be correct if the earth were spherical
    l = lc

    diff = float('inf')
    while diff > epsilon:
        next_l = lc + EP * math.sin(2 * l)
        diff = abs(next_l - l)
        l = next_l

    lat = rad2deg(l)

    # Altitude
    ro = AE * (1 - EP * math.sin(l) ** 2)
    h = r - ro

    return lat, long, h


def solveRangeEquationsViaNewton(sp: np.ndarray, rho_c: np.ndarray) -> Tuple[np.ndarray, float, np.ndarray]:
    """
    Solves the receiver position equation system, given the satellite position matrix sp
    and the corrected pseudorange vector rho_c.

    In an ideal situation, f=0. It means that all equations can be satisfied exactly.
    Satellites for which the corresponding f-value is large should be disregarded,
    provided that we still have at least 4 satellites.

    :param sp: the satellite position matrix of dimension numsat x 3,
        where numsat is the number of visible and correctly decoded satellites
    :param rho_c: a row vector of numsat elements
    :return: (rp, b, f), where
        rp is the receiver position (row vector),
        b is the difference between the range and the pseudorange,
        f is a row vector of numsat elements containing norm(sp-rp)-rho_c-b
    """

    # Compute number of satellites, raise an error if there are not enough satellites
    numsat = sp.shape[0]
    if numsat < 4:
        raise ValueError('rcvrpos:solveRangeEquationsVia_Newton:NotEnoughSatellites',
                         f'You need at least 4 satellites to compute the receiver position, got {numsat}')

    if rho_c.size != numsat:
        raise ValueError('rcvrpos:solveRangeEquationsVia_Newton:WrongDimension',
                         f'rho_c must have the length of number of satellites, but got {rho_c.size}')

    # Start with an initial guess
    rp = np.zeros((1, 3))
    b = 0

    # The idea is to use Newton's method for finding the x for which
    # f_k(x) = 0 for all k, where k is an index to the visible satellites,
    # x=[rp,b] and f_i(x)= norm(sp_i-rp)-rph_c(i)-b.
    # Because there are more equations than variales, and because the equations
    # are noisy, we have to settle for an x for which f_k(x) is almost zero for all k.
    # There is a side note for the details

    e = float('inf')
    norm_old = float('inf')
    e_stop = 1000

    f = np.zeros(numsat)  # f=[f_1, ... ,f_numsat];
    J = np.zeros((4, numsat))  # Jacobian

    while abs(e) > e_stop:  # stop if the norm of the solution does not change anymore
        for k in range(numsat):
            buf = sp[k] - rp
            buf_norm = np.linalg.norm(buf)

            f[k] = buf_norm - (rho_c[k] + b)     # this is f(x^n)

            # k-th column of the Jacobian
            J[:buf.size, k] = -buf.flatten() / buf_norm
            J[-1, k] = -1

        # x^{n+1} = x^n - f(x^n) J^{-1}(x^n)
        buf = np.zeros(rp.size + 1)
        buf[: rp.size] = rp
        buf[-1] = b
        next_ = buf - f.dot(np.linalg.pinv(J))
        rp = next_[:3]
        b = next_[3]

        # stop if the norm of f(x^n) no longer changes
        e = norm_old - np.linalg.norm(f)
        norm_old = np.linalg.norm(f)

    return rp, b, f


def calcE(ephdata: Ephemeris, t: float) -> float:
    """
    Obtain the satellite eccentric anomaly.

    To determine E_k, we need the formulas given in Table 20.IV (sheets 1 and 2),
    pp. 97-98 of the GPS standard document (Section 20.3.3.3.3.1)

    :param ephdata: Ephemeris data of the satellite
    :param t: the GPS time for which we want to determine DeltaT
    :return: eccentric anomaly
    """
    """TBC: To Be Completed"""

    # Load data from ephdata for easier use (less messy code)
    a_s = ephdata.sqrt_a ** 2   # Semimajor axis
    delta_n = ephdata.delta_n   # Correction for mean motion
    m_0 = ephdata.M_0           # Mean anomaly at reference time
    t_oe = ephdata.t_oe         # Ephemeris reference time
    e = ephdata.e               # Orbit ellipse eccentricity
    mu_e = gpsc.mu_e

    # Mean motion and corrected mean motion
    """TBC"""
    n_0 =
    """TBC"""
    n =

    # Compute time since reference time and limit it to the correct range
    t_k = limitValidRange(t - t_oe)

    # Mean anomaly
    """TBC"""
    m_k =

    # Iterative algorithm to obtain eccentric anomaly E_k
    # We need to solve M_k = E_k - e * sin(E_k) for E_k; since there is no
    # analytic solution we use the iterative algorithm seen in class.
    e_tolerance = 1e-14

    """TBC:STARTBLOCK"""

    """TBC:ENDBLOCK"""

    return e_k


def calcDeltaT(ephdata: Ephemeris, e_k: float, t: float) -> float:
    """
    Obtain the satellite clock offset.

    To determine the clock offset we use formulas from Sections 20.3.3.3.1 and 20.3.3.3.2, pages 88 and 90.

    :param ephdata: Ephemeris data of the satellite
    :param e_k: eccentric anomaly
    :param t: the GPS time for which we want to determine DeltaT
    :return: satellite clock offset
    """
    """TBC: To Be Completed"""

    # Relativistic constant used to compute the satellite clock offset
    F = -2 * math.sqrt(gpsc.mu_e) / gpsc.C ** 2

    # Load data from ephdata for easier use (less messy code)
    a_f0 = ephdata.a_f0         # 0th order,
    a_f1 = ephdata.a_f1         # 1st order, and
    a_f2 = ephdata.a_f2         # 2nd order polynomial coefficients for correction term for satellite clock offset
    t_oc = ephdata.t_oc         # Clock data reference time
    t_gd = ephdata.T_GD         # Group delay differential

    a_s = ephdata.sqrt_a ** 2   # Semimajor axis
    e = ephdata.e               # Orbit ellipse eccentricity

    # Relativistic correction term (IS-GPS-200D 20.3.3.3.1, p.88)
    # (Not to be confused with the receiver clock bias which is also called Delta_t_r in the lecture)
    """TBC"""
    delta_t_r =

    # Time since clock error reference time; corrected to be in the range [-302400, 302400]
    dt = limitValidRange(t - t_oc)

    # Compute satellite clock offset before L1 correction
    # (IS-GPS-200D, 20.3.3.3.1, p.88)
    """TBC"""
    delta_t_sv =

    # The next line concerns a detail not discussed in the notes.
    # There are actually two code sequences, of different lengths, denoted L1 and L2.
    # We are using only the L1 code. (The L2 was originally designed for military purposes.)
    # The correction that follows is needed for the L1 sequence.
    # (IS-GPS-200D, 20.3.3.3.3.2, p.90)
    Delta_t_SV_L1PY = delta_t_sv - t_gd
    return Delta_t_SV_L1PY


def satpos(ephdata: Ephemeris, t: float) -> np.ndarray:
    """
    Compute satellite position in ECEF coordinate system at GPS time t.

    To implement this function we need the formulas given in Table 20.IV
    (sheets 1 and 2), pp. 97-98 of the GPS standard document (Section 20.3.3.3.3.1)

    :param ephdata: computations use multiple parameters available in this ephemeris structure
    :param t: GPS time
    :return: a vector of length 3 of (x,y,z) coordinates in the ECEF system corresponding to GPS time t.
    """
    """TBC: To Be Completed"""

    mp.dps = 1_000_000_000_000_000  # set the precision of 1e-15

    omega = ephdata.w               # Argument of perigee
    a_s = ephdata.sqrt_a ** 2       # Semimajor axis
    t_oe = ephdata.t_oe             # Ephemeris reference time
    e = ephdata.e                   # Orbit ellipse eccentricity

    c_us = ephdata.C_us
    c_rs = ephdata.C_rs
    c_is = ephdata.C_is
    c_uc = ephdata.C_uc
    c_rc = ephdata.C_rc
    c_ic = ephdata.C_ic             # computed inclination angle
    idot = ephdata.idot             # Rate of change of the inclination angle
    omegadot = ephdata.Omegadot     # Rate of change of the right ascension
    omega_0 = ephdata.Omega_0       # Longitude of the ascending node at reference time
    i_0 = ephdata.i_0               # Inclination angle at reference time

    omega_dot_e = gpsc.Omega_dot_e  # Earth's rotational rate [rad/s]

    # Determine the satellite eccentric anomaly at time t
    e_k = calcE(ephdata, t)

    # Time since ephemeris reference time; corrected to be in the range
    # [-302400, 302400] (IS-GPS-200D Table 20-IV, p.97)
    t_k = limitValidRange(t - t_oe)

    # True anomaly
    """TBC"""
    nu_k =

    # Argument of latitude before correction
    """TBC"""
    phi_k =

    # Second harmonic perturbation corrections
    cos_2phi = mp.cos(2 * phi_k)
    sin_2phi = mp.sin(2 * phi_k)
    """TBC"""
    delta_u_k =
    """TBC"""
    delta_r_k =
    """TBC"""
    delta_i_k =

    # Corrected argument of latitude
    """TBC"""
    u_k =

    # Corrected orbital plane inclination
    """TBC"""
    i_k =

    # Corrected radius
    """TBC"""
    r_k =

    # Angle for ECEF conversion
    """TBC"""
    omega_k =
    # The above is the formula as written in (IS-GPS-200D Table 20-IV, p.98).
    # We get the same result up to a multiple of 2*pi if we use
    # the expression given in the lecture notes, namely
    # Omega_k = Omega_0 + Omegadot * t_k - Omega_dot_e * (t_k + t_oe);

    # Compute (x,y) position in orbit plane
    """TBC"""
    x_k_prim =
    """TBC"""
    y_k_prim =

    # Convert to ECEF coordinates
    """TBC"""
    x_k =
    """TBC"""
    y_k =
    """TBC"""
    z_k =

    return np.array([x_k, y_k, z_k], dtype=float)


def rotate_z(p: np.ndarray, phi: float) -> np.ndarray:
    """
    Rotate around z axis

    This function rotates the point P (in some coordinate system) around
    the z axis in the positive direction by an angle PHI [rad]. Both P and Q are row vectors, PHI is a scalar
    """

    # Create rotation matrix
    

    return R.dot(p.T)

