import numpy as np
import scipy.io as sio
from typing import Tuple, List, Union
import os
import time

import os.path
import my_utilGpsPosition as ugp

import gpsc as gpsc

from sys import path
path.append('../ephemerides/')
from utilGpsEphemerides import Ephemeris


def rcvrpos(sat: Union[List, Tuple, np.ndarray] = None) -> Tuple[float, float, float]:
    """
    Compute GPS receiver position in WGS84 coordinate system.

    :param sat: a vector of at least four satellite numbers whose data is to be used.
        If not specified, a default set of satellites is loaded from
        the list found in the file <gpsc.datadir>/correct_sats.mat
    :return: (lat, long, h) - the latitude, longitude and
        altitude in the WGS-84 geodetic system of a GPS receiver.
    """

    """TBC: To Be Completed"""

    # Set default satellites if not specified
    if sat is None:
        correct = sio.loadmat(os.path.join(gpsc.datadir, 'correct_sats.mat'))
        sat = correct['correct_sats'].flatten()

    try:
        if sat.size < 4:
            raise ValueError('rcvrpos:solve_rcvr_eq:NotEnoughSatellites',
                             f'You need at least 4 satellites to compute the receiver position, but got {sat.size}')
    except AttributeError:
        if len(sat) < 4:
            raise ValueError('rcvrpos:solve_rcvr_eq:NotEnoughSatellites',
                             f'You need at least four satellites to compute the receiver position, but got {len(sat)}')

    # here starts the actual work
    sp = np.zeros((len(sat), 3))  # Pre-allocation for satellite positions
    rho_c = np.zeros(len(sat))  # Pre-allocation for corrected pseudoranges

    # consider one satellite at a time
    for k, cur_sat in enumerate(sat):
        # Load ephemeris and pseudorange for k-th visible satellite
        # for the memory: the pseudorange rho is a(1+nu)c
        filename = os.path.join(gpsc.datadir, f'ephemerisAndPseudorange{cur_sat:02d}.mat')
        aux = sio.loadmat(filename)
        rho: float = aux['pseudorange'][0][0]
        ephemeris = Ephemeris()  # initialize the structure
        ephemeris.load(filename)  # fill the fields with the values from the file

        # Get the time t_tr when the start of the reference subframe is supposed to be transmitted
        t_tr = ephemeris.t_tr

        # Compute satellite clock offset
        # Since delta_t_s depends on E, and in turn we want to have E at a
        # time that depends on delta_t_s, we compute delta_t_s and E iteratively
        delta_t_s = 0   # our initial estimate of the offset
        _time = t_tr - rho / gpsc.C     # sat clock when signal received at t* is transmitted
        t = _time - delta_t_s   # initial estimate of GPS time that corresponds to above time.

        # The following loop updates the GPS time at each iteration considering the new delta_t_s
        # Stop when delta_t_s varies by less than 10^(-10)
        delta_t_s_tol = 1e-10
        while True:
            last_delta_ts = delta_t_s
            E_k = ugp.calcE(ephemeris, t)
            delta_t_s = ugp.calcDeltaT(ephemeris, E_k, t)
            t = _time - delta_t_s   # correct time by satellite clock offset to get GPS time
            if abs(delta_t_s - last_delta_ts) < delta_t_s_tol:
                # desired accuracy reached
                break

        # Determine the corrected pseudorange from the pseudorange and the satellite clock offset
        """TBC"""
        rho_c[k] =

        # Compute sat position at GPS time t_tr - delta_t_s - a(1+nu) = t_tr - rho_c/c,
        # in reference system ECEF(t_tr - rho_c/c)
        # (Step 5 in lecture notes)
        sp[k, :] = ugp.satpos(ephemeris, t_tr - rho_c[k] / gpsc.C)

        # Determine the sat postion in the same ECEF for all satellites, namely ECEF(t_tr)
        """TBC"""
        sp[k, :] =

    # Compute receiver position in ECEF(t_tr) and the b, where
    # b is the difference between the range and the corrected pseudorange
    # (see notes, step 6)
    rp_ecef, b, f = ugp.solveRangeEquationsViaNewton(sp, rho_c)     # Newton's method

    # check f and keep only the 4 satellites that have the smallest f
    indices = np.argsort(np.abs(f))
    selected_sats = indices[:4]  # 4 satellites that give the most consistent results
    rp_ecef, b, f = ugp.solveRangeEquationsViaNewton(sp[selected_sats, :], rho_c[selected_sats])

    # At this point we have the receiver position at receiver time t^* in ECEF(t_tr).
    # The receiver time t^* is the GPS time b/c+t_tr.
    # We want the same position in ECEF(b/c+t_tr)
    # (see notes, step 7)
    """TBC"""
    rp_ecef =

    # Convert to WGS84
    # (see notes, step 8)
    return ugp.ecef2wgs84(rp_ecef[0], rp_ecef[1], rp_ecef[2])


if __name__ == '__main__':
    # start = time.time()
    latitude, longitude, height = rcvrpos()

    print(f'Received latitude: {latitude:.06f}')
    print(f'Received longitude: {longitude:.06f}')
    print(f'Received height: {height:.06f}')

    # This prints a link to googlemap
    url = f'http://maps.google.com/maps?q={latitude:.06f},{longitude:.06f}'
    print(f'Link to Google Maps: {url}')

    # print(time.time() - start)
