function [lat, long, h] = sol_rcvrpos(sat)
%RCVRPOS   Compute GPS receiver position in WGS84 coordinate system
%   [LAT, LONG, H] = RCVRPOS(SAT) returns the latitude, longitude and
%   altitude in the WGS-84 geodetic system of a GPS receiver. SAT is a
%   vector of at least four satellite numbers whose data is to be used. 
%   If SAT is not specified, a default set of satellites is loaded from
%   the list found in the file <gpsc.datadir>/correct_sats.mat

%   TBC: To Be Completed

global gpsc; % declare gpsc as global, so we can access to it

% if gpsc has not been initialized yet, do it
if isempty(gpsc)
    gpsConfig();
end

% initializes function handles
function_mapper; 

% Set default satellites if not specified
if (nargin < 1)
    correct = load(fullfile(gpsc.datadir, 'correct_sats.mat'));
    sat = correct.correct_sats;
else
    if numel(sat)<4
        error('rcvrpos:solve_rcvr_eq:NotEnoughSatellites', 'You need at least four satellites to compute the receiver position');
    end
end

% here starts the actual work
sp = zeros(length(sat), 3); % Pre-allocation for satellite positions
rho_c = zeros(1, length(sat)); % Pre-allocation for corrected pseudoranges

% consider one satellite at a time
for k = 1:length(sat)
    
    % Load ephemeris and pseudorange for k-th visible satellite
    % for the memory: the pseudorange rho is a(1+nu)c
    filename = fullfile(gpsc.datadir, sprintf('ephemerisAndPseudorange%02d.mat', sat(k)));
    aux = load(filename);
    ephemeris = aux.ephemeris;
    rho = aux.pseudorange;
    
    % Get the time t_tr when the start of the reference subframe is supposed to be transmitted 
    t_tr = ephemeris.t_tr; 

    % Compute satellite clock offset 
    % Since delta_t_s depends on E, and in turn we want to have E at a
    % time that depends on delta_t_s, we compute delta_t_s and E iteratively   
    delta_t_s = 0;      % our initial estimate of the offset
    time = t_tr-rho/gpsc.C; % sat clock when signal received at t* is transmitted 
    t = time - delta_t_s; % initial estimate of GPS time that corresponds to above time. 
    
    % The following loop updates the GPS time at each iteration considering the new delta_t_s
    % Stop when delta_t_s varies by less than 10^(-10)
    delta_t_s_tol = 1e-10;
    while true
        last_delta_ts = delta_t_s;
        E_k = calcE(ephemeris, t);
        delta_t_s = calcDeltaT(ephemeris, E_k, t);    	            
        t = time - delta_t_s; % correct time by satellite clock offset to get GPS time        
        if (abs(delta_t_s - last_delta_ts) < delta_t_s_tol)
            break; % desired accuracy reached
        end        
    end  

    % Determine the corrected pseudorange from the pseudorange and the satellite clock offset
    rho_c(k) = rho + delta_t_s * gpsc.C; % TBC
    
    % Compute sat position at GPS time t_tr - delta_t_s - a(1+nu) = t_tr - rho_c/c,
    % in reference system ECEF(t_tr - rho_c/c)
    % (Step 5 in lecture notes)
    sp(k, :) = satpos(ephemeris, t_tr-rho_c(k)/gpsc.C);
    
    % Determine the sat postion in the same ECEF for all satellites, 
    % namely ECEF(t_tr)
    sp(k, :) = rotate_z(sp(k, :), gpsc.Omega_dot_e * rho_c(k)/gpsc.C); % TBC
    
end % for k = 1:length(sat)

% Compute receiver position in ECEF(t_tr) and the b, where
% b is the difference between the range and the corrected pseudorange 
% (see notes, step 6)
[rp_ecef, b, f] = solveRangeEquationsVia_Newton(sp, rho_c); % Newton's method
%[rp_ecef, b, f] = solveRangeEquationsVia_fsolve(sp, rho_c) % Matlab's fsolve
 
 %  check f and keep only the 4 satellites that have the smallest f
 [~,indices] = sort(abs(f));
 selectedSats = indices(1:4); % % 4 satellites that give the most consistent results 
 [rp_ecef, b, f] = solveRangeEquationsVia_Newton(sp(selectedSats,:), rho_c(selectedSats));

% At this point we have the receiver position at receiver time t^* in ECEF(t_tr).
% The receiver time t^* is the GPS time b/c+t_tr.
% We want the same position in ECEF(b/c+t_tr)
% (see notes, step 7)
rp_ecef = rotate_z(rp_ecef, gpsc.Omega_dot_e * b/gpsc.C); % TBC

% Convert to WGS84
% (see notes, step 8)
[long, lat, h] = ecef2wgs84(rp_ecef(1), rp_ecef(2), rp_ecef(3));

end % function rcvrpos()
