function [s, ind] = sol_removeExcessBits(bits)
%REMOVEEXCESSBITS Extracts the complete subframes in the bit sequence
%obtained from a satellite
%   [S, IDX] = REMOVEEXCESSBITS(BITS) detects the start of the first
%   subframe in the row vector BITS (values in {-1,+1}) by correlating
%   with the GPS preamble (stored in gpsc.preamble as a sequence of
%   1s and 0s). If the negative of the preamble is found all bits
%   are inverted. IDX is the index into BITS where the first subframe
%   starts; incomplete subframes at the beginning and at the end of
%   BITS are removed and the sequence of bits is converted into a
%   sequence of {0,1} values ({1,-1} <-> {0,1}) and returned into
%   vector S.
%
% Hints:
% - The preamble is stored in gpsc.preamble as a sequence of 0s and 1s.
% - It is not enough to just find one preamble: the preamble is 8 bits
%   long and it is not unlikely that this 8-bit sequence is also
%   present somewhere in the middle of the data sequence. You should
%   use the fact that there is one preamble in every subframe (every
%   300 bits).
% - The result of xcorr() are real numbers and it can have multiple maxima.
%   If you want to compare two values returned by xcorr(), you should first
%   round them using MATLAB's function round().


% declare gpsc as global, and if it is not initialized, do it
global gpsc;
if isempty(gpsc)
    gpsConfig();
end


% The preamble to find (converted to BPSK, {0,1} <-> {1, -1})
pa = 1 - 2 * gpsc.preamble;

% Minimum number of subframes to extract (can vary  +1)
nsf = floor(length(bits) / gpsc.bpsf) - 1;

% Check that we have sufficient data to correctly identify the preamble
if nsf < 2
    error('removeExcessBits: not enough input bits');
end


% Create the repetition of preamble for correlation
pa_r = repmat([pa zeros(1, gpsc.bpsf - length(pa))], 1, nsf);

% Compute the correlation between the bit sequence and the replicated
% preamble. The maximum value will indicate the start of a subframe.
% We round the output of xcorr to remove rounding errors. Then we take the
% first maxima (there can be several of them).
c = round(xcorr(bits, pa_r));
ind = find(abs(c) == max(abs(c)));
ind = ind(1);

% Convert bits to 0s and 1s according to the maximum correlation value:
% if it is positive, then the mapping is {1, -1} <-> {0, 1}
% if it is negative, then the mapping is {1, -1} <-> {1, 0}

if c(ind) < 0
    bits = double(bits > 0);
else
    bits = double(bits < 0);
end


% Correct index to point to the right position of bits
ind = ind - length(bits) + 1;

% Nicolae: if index is out of range, throw error.
if (ind < 1) || (ind > length(bits))
    error('LCM:NoStartSubframe', 'removeExcessBits: could not find the start of a subframe, probably wrong bits...\n');
end


% Strip off incomplete subframes at the beginning
s = bits(ind:end);

% Strip off incomplete subframes at the end
excessEnd = mod(length(s), gpsc.bpsf);
s = s(1:end - excessEnd);

end
