import requests
from typing import Dict, Any
from agents import Agent, function_tool


@function_tool
def get_food_by_barcode(barcode: str) -> Dict[str, Any]:
    """
    Retrieves food information from OpenFoodFacts API by barcode.

    Args:
        barcode: The barcode number of the food product

    Returns:
        Dictionary containing food information if found, or error message if not found
    """
    url = f"https://world.openfoodfacts.org/api/v0/product/{barcode}.json"

    try:
        response = requests.get(url)
        response.raise_for_status()  # Raise an exception for 4XX/5XX responses

        data = response.json()

        if data.get("status") == 0:
            return {"error": "Product not found", "status": data.get("status")}

        # Extract only relevant information
        product = data.get("product", {})

        # Compile the most important product details
        result = {
            "product_name": product.get("product_name"),
            "brands": product.get("brands"),
            "image_url": product.get("image_url"),
            "nutriscore_grade": product.get("nutriscore_grade"),
            "nutriments": product.get("nutriments", {}),
            "categories": product.get("categories"),
            "ingredients_text": product.get("ingredients_text"),
            "allergens": product.get("allergens"),
            "labels": product.get("labels"),
            "serving_size": product.get("serving_size"),
            "nova_group": product.get("nova_group"),
            "ecoscore_grade": product.get("ecoscore_grade"),
            "source": "OpenFoodFacts",
        }

        return result

    except requests.exceptions.RequestException as e:
        return {"error": f"API request error: {str(e)}"}


openfoodfacts_agent = Agent(
    name="OpenFoodFacts Agent",
    instructions="""You are an agent that retrieves food information from the OpenFoodFacts database.
    
    When a user provides a barcode, use the get_food_by_barcode function to fetch information about that product.
    
    If the product is found:
    - Provide a concise summary of the product
    - Include key nutritional information
    - Mention relevant health/nutrition scores if available
    
    If the product is not found:
    - Let the user know the product wasn't found in the database
    - Ask if they'd like to try another barcode

    You don't need to perform any calculations on the data - just present it in a helpful, readable format.
    """,
    handoff_description="Specialist agent for looking up food products by barcode in the OpenFoodFacts database",
    tools=[get_food_by_barcode],
)
