#!/bin/bash
# ********************
#
# Script to execute a generic 3D non-linear dynamics C++ code for a range of input parameters and write the results to a series of data files
# that can be plotted using gnuplot and illustrate the effects of parameters on the limit cycle in the model.
#
#
# Preconditions to this script
# *****************************************************
#
#  All paths and filenames are hardwired.
#
# Version History
# ***************
#
#  12/06/24  1.0  Author: Julian Shillcock. Generic nonlinear 2D system
#  24/06/24  1.1  Modified to match dx/dt = mu - x**2, dy/dt = -y.
#
#
# Arguments
# *********
#
#  $1      = Number of steps in the integration
#  $2      = mu
#  $3      = lower X axis limit
#  $4      = upper X axis limit
#  $5      = lower Y axis limit
#  $6      = upper Y axis limit
#  $7      = lower Z axis limit
#  $8      = upper Z axis limit
#
# ******************************************************************************************

if [ $# != 8 ]; then echo "****** Insufficent arguments to script (supply n, mu, xlow, xhigh, ylow, yhigh, zlow, zhigh), aborting ******"; exit ; fi

# Set the unchanging parameters

prefix=nonlin3
inputName=${prefix}input
dataName=${prefix}data
executable=$prefix

# Model parameters and number of steps

n=$1
mu=$2

# Axis limits of the plot

xlow=$3
xhigh=$4
ylow=$5
yhigh=$6
zlow=$7
zhigh=86

 

# ********************
# Iterate over a set of initial points giving each one a different filename. It currently just appends the points
# to the filename, which works only for legal characters, but . and - are ok.

#yvalues="-8 -4 2 -1.5 -1.1 -1 -0.9 -0.75 -0.5 0.5 0.75 0.9 1 1.1 1.5 2 4 8"

#xvalues="-1 -2 -3 -4 -5 -6 -8 -10 1 2 3 4 5 6 8 10"
#yvalues="-1 -2 -3 -4 -5 -6 -8 -10 1 2 3 4 5 6 8 10"
#zvalues="-1 -2 -3 -4 -5 -6 -8 -10 1 2 3 4 5 6 8 10"

#xvalues="-10 -1 1 10"
#yvalues="-10 -1 1 10"
#zvalues="-10 -1 1 10"

xvalues="-1 1"
yvalues="-1 1"
zvalues="-1 1"

# Make a list of all the data files to add the plot script

dataList=""

for x in $xvalues; do
for y in $yvalues; do
for z in $zvalues; do

if [ -e $inputName ]; then rm $inputName; fi

echo ${dataName}$x$y$z $n $mu $x $y $z > $inputName

$executable < $inputName >> /dev/null

dataList=$dataList" "${dataName}$x$y$z

done
done
done

# Plot the results in gnuplot, using the current date/time to identify the plot. We use sed/awk to change the name
# of the plot file and other variables depending on the curve being plotted.

current_time=$(date "+%Y.%m.%d-%H.%M.%S")

# Set the name of the output file

sed "s/prefix/$prefix/g;s/currentTime/$current_time/" $prefix.txt > ${prefix}1.txt

# Set the limits of the plot

sed "s/limits/[$xlow:$xhigh][$ylow:$yhigh]/" ${prefix}1.txt > ${prefix}2.txt

# Add the parameter value as a label in the bottom left  corner

sed "s/xlow/$xlow/; s/yhigh/$yhigh/;  s/aValue/$mu/; " ${prefix}2.txt > ${prefix}3.txt

# Insert the names of all the relevant data files into the plot file

for item in $dataList; do
    awk -v name=$item '{ if($1=="dataLine") {print "\""name"\"" " using 1:2:3 with lines,\\\n dataLine"} else {print}}' ${prefix}3.txt > ${prefix}2.txt
    
    mv ${prefix}2.txt ${prefix}3.txt
done


awk -v name=$item '{ if($1=="dataLine") {print "\""name"\"" " using 1:2:3 with lines \n"} else {print}}' ${prefix}3.txt > ${prefix}2.txt

# Make a plot with no legend so the picture is clear

cp ${prefix}2.txt ${prefix}3.txt

gnuplot ${prefix}3.txt

# Clean up by deleting intermediate files

rm ${prefix}1.txt
rm ${prefix}2.txt
rm ${prefix}3.txt

rm $inputName
rm ${dataName}*
