%% Tutorial for time-varying geometry simulation
% Simulation with time-varying geometry is demonstrated in case of plasma
% elongation increase from 1.1 to 1.7. 
% An increase in the edge safety factor q95 is expected. 
% Two CHEASE data files are used to get equilibrium data for low and high
% plasma elongation.

%% Model construction and initialization
close all hidden;
run(fullfile(pwd,'..','RAPTOR_path.m'));
% Load default parameters
config = RAPTOR_config; 
% Specify input data file(s)
% One can specify several input data file(s) corresponding to plasma
% equilibrium at different time steps. In case of several data
% files config.equi.filenames contains a cell array.
config.equi.filenames = [{'/tutorial_freegeom_t0.1_cheasedata.mat'},...
           {'/tutorial_freegeom_t0.4_cheasedata.mat'}];
% Specify time grid for input data
config.equi.times = [0.1 0.4];

% Define time grid: time interval from 0.0 to 450.0 ms
config.grid.tgrid = [0:0.005:0.45]; 

% Build model and get params, init structures, geometry g, kinetic profiles
% v, actuator inputs U
[model,params,init,g,v,U] = build_RAPTOR_model(config);

% Define initial conditions for plasma state
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1)); 

%% Run RAPTOR simulation   
simres = RAPTOR_predictive(x0,g,v,U,model,params);
out = RAPTOR_out(simres,model,params);

%% Plot results
subplot(211)
plot(out.time,[out.q95;out.q(end,:);out.Ip(end,:)/1e4]);
xlabel('time, sec'); legend({'q_{95}','q_{edge}','Ip [10kA]'},'location','northwest');
subplot(212)
plot(out.time,[out.te(1,:);out.Wth]);
xlabel('time, sec'); legend({'Te0','W_{e}'},'location','northwest');

