classdef liux_test < meq_test
 % Test various sources of data for liuqe
 %
 % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
 
 properties
  shot = 65488
  t = -0.4:1e-2:1;
  verbosity = 0;
 end
 
 properties(TestParameter)
   historicshot = struct('first_liuqe',11644,'no_UFOS',40000,'recent',68888,...
                         ... % TCV trophy shots
                         'limited',9849,...
                         'SN_upper',6010,...
                         'SN_lower',5650,...
                         'DN',8856,...
                         'Highest_squareness',11962,...
                         'Highest_current',11368,...
                         'Highest_fully_ECCD_driven_current',18548,...
                         'Pear_shape',6442,...
                         'Doublet_shape',10159,...
                         'Highest_elongation',19373,...
                         'Lowest_triangularity',11928,...
                         'Snowflake',36151); 
   rtcshot = {'57587','65488','70392'};
 end
 
 methods (Test, TestTags={'TCV'})
  
  function test_liuxrtc(testCase,rtcshot)
   myshot = str2double(rtcshot);
   [L,LX] = liu('tcv',myshot,testCase.t);

   LXrt = liuxrtc(myshot,testCase.t,L); % get defaults
   
   % compare
   if testCase.verbosity
    liuxcomp(LX,LXrt,{'offline','rt'});
   end
  end
  
  function test_timebase(testCase)
   [L,LX,LY] = liuqe(testCase.shot,testCase.t);
   
   % Verify that input times correspond to a true measurement
   tmag = mdsvalue('\bpol_003:dim0');
   testCase.verifyTrue(all(ismember(LX.t,tmag)),'Some values in LX.t do not correspond to any value in \\bpol_003:dim0');
   
   % Verify that output times are input times
   testCase.verifyTrue(all(ismember(LY.t,LX.t)),'Some values in LY.t do not correspond to any value in LX.t');
   
   % Verify that measurements can be recovered from subscripting input signals with output times.
   conv = ismember(LX.t,LY.t);
   bm = mdsvalue('\bpol_003[$1,*]',mdscvt(LY.t,'f'));
   testCase.verifyEqual(LX.Bm(:,conv),bm.','Cannot recover LX.Bm values by subscripting input signals');
  end
  
  function test_historic(testCase,historicshot)
    % Test retrieval of all quantities across TCV epochs
    time = 0:1e-2:2;
    % Restrict to time with magnetics data
    [~,st] = mdsopen('tcv_shot',historicshot);
    assert(isodd(st),'Could not open shot %05d',historicshot);
    td = mdsvalue('\magnetics::iphi:dim0');
    time = time(time >= td(1) & time <= td(end));
    %
    kalman = (historicshot>=49830);
    [L,LX] = liuqe(historicshot,time,'kalman',kalman);
    testCase.verifyEqual(LX.t,time,'AbsTol',1e-6);
  end
 end
 
end
