classdef interpolators_test < meq_test
 % Test various flux/field interpolators in liut.m
 %
 % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

 properties
  tolF = 5e-3; % error tolerance on F interpolation
  infct; % interpolator function handle
  inp; % interpolator parameters
  L,LX; % LIUQE structures
 end
 
 properties(TestParameter)
   rn = struct('empty',[],'three',[1;1;1])
   zn = struct('empty',[],'three',[-0.2;0;0.2]);
 end
 
 properties(MethodSetupParameter)
   % interpolator: function handle for interpolator
   % intp: parameter structure for interpolator
   interpolator = struct('none',[],'quadratic_6p',    'qintmex','quadratic_9p','qintmex'    );
   interpparams = struct('none',[],'quadratic_6p',struct('n',6),'quadratic_9p',struct('n',9));
 end
 
 methods(TestClassSetup)
   function setup_LLX(testCase)
     testCase.assumeFalse(isempty(which('mdsopen')),'Skip test since no MDS found')
     tok = 'ana'; shot = 1;
     [Lb,~,LYb] = fbt(tok,shot);
     testCase.L = liu('ana',shot);
     testCase.LX = meqxconvert(Lb,LYb,testCase.L);
   end
 end
 
 methods(TestMethodSetup)
   
   function set_infct(testCase,interpolator,interpparams)
     % sets up function handle for interpolator to be tested
     if ~isempty(interpolator)
       testCase.infct = str2func(interpolator);
     else
       testCase.infct = []; % also test 'empty' case (no interpolation)
     end
     testCase.inp = interpparams;
   end
 end
 
 methods(Test,TestTags={'Unit'}, ParameterCombination='sequential' )
   
  function test_liuqe(testCase,rn,zn)
   
   L = testCase.L; %#ok<*PROPLC>
   LX = testCase.LX;

   % recompute consolidation with this intepolator
   L = liuc(L.P,L.G,...
     'infct',testCase.infct,'inp',testCase.inp,...
     'rn',rn,'zn',zn); 
   
   % assign on which ri,zi to give outputs
   LY = liut(L,LX);
   
   if isempty(testCase.infct) || isempty(rn)
     testCase.assertTrue(~isfield(LY,'Fn' ),'Fn  should be absent from LY')
     testCase.assertTrue(~isfield(LY,'Brn'),'Brn should be absent from LY')
     testCase.assertTrue(~isfield(LY,'Bzn'),'Bzn should be absent from LY')
   else
     Fxi = interp2(L.rx,L.zx,LY.Fx,L.P.rn,L.P.zn); % matlab interpolation
     testCase.assertEqual(LY.Fn,Fxi,'RelTol',testCase.tolF)
   end
  end
 end
end
