classdef bfef_tests < meq_test
  % Compare bfefmex with bfef
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    tol = 100*eps;
    verbosity = 0;
    n = 41; % number of FN points

    % parameters for dummy test equilibrium
    r0 = 1;
    rBt = 1.4;
    
    FN; L; LY;
  end
  
  properties (ClassSetupParameter)
    % Various combinations of boundary and axis flux
    FAs    = {-1   0   2  0.5 };
    FBs    = {0    1   0  0   };
  end
  
  properties (TestParameter)
    bfp = {[2,3],[0,2],[1,0]}
  end
  
  methods(TestClassSetup,ParameterCombination='sequential')
    
    function gen_Fx(testCase,FAs,FBs)
      r0 = testCase.r0; z0=0; rBt = testCase.rBt; %#ok<*PROPLC>
      L = liu('ana',1);
      
      [testCase.L,testCase.LY] = testCase.getCircularEquilibrium(L,r0,z0,FAs,FBs,rBt);
      testCase.FN = linspace(0,1,testCase.n)';
     end
    
  end

  methods(Test, TestTags = {'Unit','bf'})
   
    function test_mode0(testCase,bfp)
      % Test 
      [fPg ,fTg ] = bfefmex(0,bfp);
      [fPg_,fTg_] = bfef(0,bfp);
      testCase.verifyEqual(fPg,fPg_,'mismatch for fPg');
      testCase.verifyEqual(fTg,fTg_,'mismatch for fTg');
    end
    
    function test_mode1(testCase,bfp)
      % Test evaluating basis function and its integrals on rz grid
      L  = testCase.L ;
      LY = testCase.LY;
      [Tyg ,TpDg ,ITpDg ] = bfefmex(1,bfp, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      [Tyg_,TpDg_,ITpDg_] = bfef(1,bfp, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      
      % Verify match
      testCase.verifyEqual(Tyg  , Tyg_  ,'RelTol',testCase.tol,'mismatch for Tyg')
      testCase.verifyEqual(TpDg , TpDg_ ,'RelTol',testCase.tol,'mismatch for TpDg')
      testCase.verifyEqual(ITpDg, ITpDg_,'RelTol',testCase.tol,'mismatch for ITpDg')
      
    end
    
    function test_mode2(testCase,bfp)
      % Test evaluating basis function and its integrals on FN grid
      LY = testCase.LY;
      [GN ,IGN ] = bfefmex(2,bfp, ...
        testCase.FN,LY.FA,LY.FB);
      [GN_,IGN_] = bfef(2,bfp, ...
        testCase.FN,LY.FA,LY.FB);
      
      % Verify match
      testCase.verifyEqual(GN , GN_ ,'RelTol',testCase.tol,'mismatch for GN')
      testCase.verifyEqual(IGN, IGN_,'RelTol',testCase.tol,'mismatch for IGN')
      
    end
    
    function test_mode3(testCase,bfp)
      % Test transformation of coefficients for profile evaluation
      L  = testCase.L ;
      LY = testCase.LY;
      [fPg,fTg] = bfefmex(0,bfp);
      ag0 = ones(size(fPg)); % unit coefficients just to get scaling
      
      [aPpg ,aTTpg ,aPg ,ahqTg ] = bfefmex(3,bfp,ag0,...
        LY.FA,LY.FB,fPg,fTg,L.idsx);
      [aPpg_,aTTpg_,aPg_,ahqTg_] = bfef(3,bfp,ag0,...
        LY.FA,LY.FB,fPg,fTg,L.idsx);
      
      % Verify match
      testCase.verifyEqual(aPpg , aPpg_ ,'RelTol',testCase.tol,'mismatch for aPPg')
      testCase.verifyEqual(aTTpg, aTTpg_,'RelTol',testCase.tol,'mismatch for aTTpg')
      testCase.verifyEqual(aPg  , aPg_  ,'RelTol',testCase.tol,'mismatch for aPg')
      testCase.verifyEqual(ahqTg, ahqTg_,'RelTol',testCase.tol,'mismatch for ahqTg')
      
    end
    
    function test_mode5(testCase,bfp)
      % Test axis values
      LY = testCase.LY;
      [GA ,IGA ] = bfefmex(5,bfp,[],LY.FA,LY.FB);
      [GA_,IGA_] = bfef(5,bfp,[],LY.FA,LY.FB);
      
      % Verify match
      testCase.verifyEqual(GA , GA_ ,'RelTol',testCase.tol,'mismatch for GA')
      testCase.verifyEqual(IGA, IGA_,'RelTol',testCase.tol,'mismatch for IGA')
    end
   
    function test_mode6(testCase,bfp)
      % Test computation of elements for the regularisation equations
      L  = testCase.L ;
      LY = testCase.LY;
      rA = testCase.r0;
      args = {[],LY.FA,LY.FB,rA,1./rA,L.dsx}; % arguments common to all bfs
      [Qqg ,Xq ] = bfefmex(6,bfp,args{:});
      [Qqg_,Xq_] = bfef(6,bfp,args{:});
      
      % Verify match
      testCase.verifyEqual(Qqg, Qqg_,'RelTol',testCase.tol,'mismatch for Qqg')
      testCase.verifyEqual(Xq , Xq_ ,'RelTol',testCase.tol,'mismatch for Xq')
    end
  end
end