%IATAMEX  LDL matrix inversion
%
% This is the MATLAB equivalent implementation of libmeq/iata.c.
% A more detailed help is available in IATAMEX.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function b = iatamexm(a1,a2)
 switch size(a1,2)
  case 0, b = zeros(0,0);
  case 1, b = iata1(a1,a2);
  case 2, b = iata2(a1,a2);
  case 3, b = iata3(a1,a2);
  case 4, b = iata4(a1,a2);
  otherwise, b = iatan(a1,a2);
 end
end

function b = iata1(a1,a2)

 % [a1;a2]'*[a1;a2]
 if nargin > 1 && ~isempty(a2)
  a = a1'*a1 + a2'*a2;
 else
  a = a1'*a1;
 end
 b = 1./a;
end

function b = iata2(a1,a2)

 % [a1;a2]'*[a1;a2]
 if nargin > 1 && ~isempty(a2)
  a = tril(a1'*a1 + a2'*a2);
 else
  a = tril(a1'*a1);
 end
 a00 = a(1,1);
 a10 = a(2,1);
 a11 = a(2,2);
 
 % LDL' decomposition
 if a00, id0 = 1.0 / a00; else, id0 = 0; end
 l10 = a10 * id0;
 d1 = a11 - l10^2*a00;
 if  d1, id1 = 1.0 /  d1; else, id1 = 0; end
 
 % inversion by back substitution  inverse is symetric
 b = zeros(2);
 b(2,1) = -l10*id1;         b(1,2) = b(2,1);
 b(2,2) = id1;
 b(1,1) = id0 - l10*b(2,1);
end

function b = iata3(a1,a2)
 
 % [a1;a2]'*[a1;a2]
 if nargin > 1 && ~isempty(a2)
  a = tril(a1'*a1 + a2'*a2);
 else
  a = tril(a1'*a1);
 end
 a00 = a(1,1);
 a10 = a(2,1);
 a20 = a(3,1);
 a11 = a(2,2);
 a21 = a(3,2);
 a22 = a(3,3);
 
 % LDL' decomposition
 if a00, id0 = 1.0 / a00; else, id0 = 0; end
 l10 = a10 * id0;
 d1 = a11 - l10^2*a00;
 if  d1, id1 = 1.0 /  d1; else, id1 = 0; end
 l20 = a20 * id0;
 l21 = ( a21 -l20*l10*a00 ) * id1;
 d2 = a22 - l20^2*a00 - l21^2*d1;
 if d2 , id2 = 1.0 /  d2; else, id2 = 0; end
 
 % inversion by back substitution  inverse is symetric
 b = zeros(3);
 b(3,1) = ( l21*l10 - l20 ) * id2; b(1,3) = b(3,1);
 b(3,2) = -l21*id2;                b(2,3) = b(3,2);
 b(3,3) = id2;
 b(2,1) = -l10*id1 - l21*b(3,1);   b(1,2) = b(2,1);
 b(2,2) = id1 - l21*b(3,2);
 b(1,1) = id0 - l10*b(2,1) - l20*b(3,1);
end

function b = iata4(a1,a2)
 
 c0 = cast(0,'like',a1);
 c1 = cast(1,'like',a1);
 
 % [a1;a2]'*[a1;a2]
 if nargin > 1 && ~isempty(a2)
  a = tril(a1'*a1 + a2'*a2);
 else
  a = tril(a1'*a1);
 end
 
 a00 = a(1,1);
 a10 = a(2,1);
 a20 = a(3,1);
 a30 = a(4,1);
 a11 = a(2,2);
 a21 = a(3,2);
 a31 = a(4,2);
 a22 = a(3,3);
 a32 = a(4,3);
 a33 = a(4,4);
 % LDL' decomposition
 if a00, id0 = c1 / a00; else, id0 = c0; end
 l10 =   a10            * id0;
 d1  =   a11 - l10*a10;
 if  d1, id1 = c1 /  d1; else, id1 = c0; end
 l20 =   a20            * id0;
 l21 = ( a21 -l20*a10 ) * id1;
 d2  =   a22 -l20*a20 - l21*l21*d1;
 if  d2, id2 = c1 /  d2; else, id2 = c0; end
 l30 =   a30                        * id0;
 l31 = ( a31 -l30*a10             ) * id1;
 l32 = ( a32 -l30*a20 -l31*l21*d1 ) * id2;
 d3  =   a33 -l30*a30 -l31*l31*d1 - l32*l32*d2;
 if  d3, id3 = c1 /  d3; else, id3 = c0; end
 % inversion by back substitution                         inverse is symetric
 b15 =                 id3                              ;
 b11 =          - l32 *id3                              ; b14 = b11;
 b07 = (l21*l32 - l31)*id3                              ; b13 = b07;
 b03 = (l20*l32 - l30)*id3 - l10*b07                    ; b12 = b03;
 b10 =                 id2 -           l32*b11          ;
 b06 =          - l21 *id2 -           l32*b07          ; b09 = b06;
 b02 = (l10*l21 - l20)*id2 -           l32*b03          ; b08 = b02;
 b05 =                 id1 -           l21*b06 - l31*b07;
 b01 =          - l10 *id1 -           l21*b02 - l31*b03; b04 = b01;
 b00 =                 id0 - l10*b01 - l20*b02 - l30*b03;
 b = [b00 b04 b08 b12;
  b01 b05 b09 b13;
  b02 b06 b10 b14;
  b03 b07 b11 b15];
end

function b = iatan(a1,a2)
 
 % [a1;a2]'*[a1;a2]
 if nargin > 1 && ~isempty(a2)
  a = tril(a1'*a1 + a2'*a2);
 else
  a = tril(a1'*a1);
 end

 [L,D] = ldl(a);
 b = L.' \ ( D \ ( L \ eye(size(a,1)) ) );
end