function dim = meqsize(L,fieldname)
% dim = meqsize(L,fieldname)
% Returns the dimension of some standard LY/LX fields
% 
% e.g. L = liu('tcv',61400,1); dim = meqsize(L,'Iv');
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

switch fieldname
  case 'Fx'
    dim = [L.nzx, L.nrx];
  case 'Fz'
    dim = [L.nzz, L.nrz];
  case {'rB','zB','FB','rA','zA','FA','qA','lX','nX','dr2FA','dz2FA','drzFA'}
    dim = [L.nD 1];
  case {'Iy','Opy'}
    dim = [L.nzy, L.nry];
  case {'IpD','zIpD','rIpD','bpD'}
    dim = [L.nD,1];
  case {'Ia','Va'}
    dim = [L.G.na,1];
  case 'Iv'
    dim = [L.G.nv,1];
  case 'Iu'
    dim = [L.G.nu,1];
  case 'Is'
    if isfield(L.G,'ns'), ns = L.G.ns;
    elseif strcmp(L.P.selu(1),'s'), ns = L.G.nu;
    else, error('can''t determine ''s'' dimension');
    end
    dim = [ns,1];
  case {'PpQ','TTpQ','PQ','TQ','iTQ','iqQ','jtorQ'}
    dim = [L.nQ,1];
  case {'PpQg','TTpQg'}
    dim = [L.nQ,L.ng];
  case {'Ip','rBt','Wk','Wp','Wt','Wt0','WN','Vp','FR','Ft','Ft0','bp','bt','mu','li','bpli2'}
    dim = [1,1];
  case {'lB','nA','nB','q95','lp','rbary','raqmin','qmin','dz','dzg','zIp','rIp'}
    dim = [1,1];
  case {'rgeom','zgeom','kappa','delta','deltau','deltal','aminor','epsilon'}
    dim = [L.nQ,1];
  case {'Q0Q','Q1Q','Q2Q','Q3Q','Q4Q','raQ'}
    dim = [L.nQ,1];
  case {'Bm','Bmdot'}
    dim = [L.G.nm,1];
  case {'Ff','Ffdot','Uf'}
    dim = [L.G.nf,1];
  case {'TpDg','ITpDg'}
    dim = [L.nD,L.ng];
  case 'ag'
    dim = [L.ng,1];
  case {'Fn','Brn','Bzn','Brrn','Bzrn','Bzzn','Brzn','Vn'}
    dim = [L.nn,1];
  case {'aW'}
    dim = [L.G.nW,L.P.nFW];
  case 'FW'
    dim = [1,L.P.nFW];
  case 'raR'
    dim = [L.nR,L.P.naR];
  case {'rS','zS'}
    dim = [L.noq,L.nS,L.nD];
  case {'niter','isconverged','res','t','shot','rst','cycle','err'}
    dim = [1,1];
  case {'Fe','Fedot','Ie'}
    dim = [L.ne,1];
  case {'aq','rq','zq'}
    dim = [L.noq,L.npq];
  case {'rX','zX','FX'}
    if isempty(L.dimw), dim = [0,1];
    else,               dim = [L.dimw{:},1];
    end
  otherwise
    error('%s dimension not known', fieldname);
end
end
