%MEQINT  Equilibrium integrals
% [IP,WK,WP,WT,WT0,WN,VP,FT,FT0,BP,BT,MU,LI,BPLI2] =
% MEQINT(FP,FT,TPDG,ITPDG,A,F,O,L,RBT). 
% Computes volume integrals and related quantities. 
% Inputs;
% FP,FT,TPDG,ITPDG come from from base function call BF(1,...), 
% A: base function coefficients, 
% F(z,r): flux map, 
% O: plasma domain,
% L: structure from LIUC, 
% RBT: vacuum R*Bt
%
% Outputs:
% IP    plasma current
% WK    kinetic energy
% WP    poloidal field energy
% WT    plasma toroidal field energy
% WT0   vacuum toroidal field energy
% WN    normalisation mu0*r0*Ip^2/4
% VP    Total enclosed plasma volume
% FT    plasma toroidal flux
% FT0   vacuum toroidal flux
% BP    poloidal beta
% BT    toroidal beta
% MU    diamagnetism
% LI    internal inductance
% BPLI2 BP+LI/2
%
% For details, see: [MEQ-redbook] 
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function [Ip,Wk,Wp,Wt,Wt0,WN,Vp,Ft,Ft0,bp,bt,mu,li,bpli2] = meqint(fPg,fTg,TpDg,ITpDg,ag,Fx,Opy,L,rBt)
 
 Ip  = sum( TpDg *ag  ,1);
 x   = sum(ITpDg.*ag.',1);
 Wk  = x*fPg*1.5;
 Wt  = x*fTg; 
 Ft  = 2e-7*Wt/rBt;
 [Wp,Ft0,Vp] = vizrmex(Fx,logical(Opy),L.ry,L.iry,rBt,L.drx,L.dzx);
 Wt0 = 2.5e6*rBt*Ft0;
 WN  = 1e-7*pi*L.P.r0*Ip^2;
 bt  = 2/3*Wk/Wt0;
 if WN==0, iWN = 0;       % bp,mu,li = 0
 else,     iWN = 1/WN;end
 bp  = 2/3*Wk*iWN;
 mu  =     Wt*iWN;
 li  =     Wp*iWN;
 bpli2 = bp + li/2;
 
end
