!
! The code performs a blocking analysis on the input data
! Output: sigma(I) and P(tB) as a function of the block transformation step
!         step and relative errors 
! (cf. the documentation)
!
! Author: Giacomo Miceli
! EPFL, November 2012  
!
program blocking
 implicit none
 integer :: i,j,k
 integer :: iblock           ! step counter for the block transformation
 integer :: nstep            ! max number of steps, N
 integer :: np               ! np = N/2^M --> M is the step of block transformation
 integer :: nb               ! nb = 2^M
 
 real :: Aens                ! ensemble average <A> ~ 1/N * sum{ A_i }
 real :: sigmaAsq            ! ensamble variance <A^2> - <A>^2 ~ 1/N * sum{ ( A_i - <A> )^2 }
 real :: sigmaBsq            ! ensemble variance for the block ~ 1/np * sum{ ( AB_i - <A> )^2 }
 real :: sigmaI
 real :: PtB, errorPtB       ! tB * sigmaB^2 / sigmaA^2 = 2 tau
 real :: c0, error
 
 real, dimension(:), allocatable :: A1,A2
 
 character(256) :: inputfile
 
 call read_input
 
 open(11,file=inputfile,status='old')
 
 allocate (A1(nstep),A2(nstep))
 
 c0=0.0
 Aens=0.0
 do i=1,nstep
   read(11,*) A1(i)
   Aens = Aens + A1(i)
 enddo
 Aens = Aens/real(nstep)
  
 do i=1,nstep
   c0 = c0 + ( A1(i) - Aens )**2
 enddo 
 sigmaAsq = c0/real(nstep)
 
 iblock=0
 PtB=1.0   ! nb * sigmaB^2/sigmaA^2 ; nb=1, sigmaB=sigmaA
 sigmaI = sqrt(sigmaAsq/real(nstep-1))
 error = 1./sqrt(2.0*real(nstep-1))
 errorPtB = sqrt(2.0/real(nstep-1))
 write(*,*) '## step of transformation, sigma_I(n), sigma error, P(tB), P(tB) error'
 write(*,*) iblock, sigmaI, sigmaI*error, PtB*0.5, PtB*0.5*errorPtB
 
 ! loop of blocking transformations
 do iblock=1,nstep
   A2=0.0
   
   ! block transformation 
   np = int( real(nstep) / 2**iblock )
   if(np .eq. 1)then
     exit
   endif
   
   ! evaluation of sigmaB^2 [ -> sigmaA^2(M) in documentation.pdf ]
   c0=0.0
   do i=1,np
     A2(i)=0.5*( A1(2*i-1) + A1(2*i) )
     c0 = c0 + ( A2(i) - Aens )**2
   enddo
   A2(np+mod(np,2))=A2(np)
   sigmaBsq = c0/real(np)
   
   nb= 2**iblock
   PtB = real(nb) * sigmaBsq / sigmaAsq
   sigmaI=sqrt(sigmaBsq/real(np-1))
   error = 1./sqrt(2.0*real(np-1))
   errorPtB= sqrt(2.0/real(np-1))
   write(*,*) iblock, sqrt(sigmaBsq/real(np-1)), sigmaI*error, PtB*0.5, PtB*0.5*errorPtB
   A1=A2
 enddo
 
!###############################################################################

 contains
 
!-------------------------------------------------------------------------------
!   READ input file
!-------------------------------------------------------------------------------

 subroutine read_input
 
 namelist/blocking/&
         nstep,inputfile
 
! default parameter 
 nstep=10000
 inputfile="output"
 
 read(*,nml=blocking,end=200,err=800)
 
 return   !  successful exit
200 continue
   write(0,*)'Read_Input: FATAL: premature end-of-file in standard input'
   stop
800 continue
   write(0,*)'Read_Input: FATAL: read error in standard input'
   stop
 
 end subroutine read_input
 
 end program
