#!/usr/bin/python3

# to place 'npart' particles on a simple cubic 
# lattice with density 'rho'


from numba import jit, prange
import math
import random
import numpy as np
import time

from MC import *


def main():

    # Parameters section begin
    # Simulation parameter
    npart=200       # Number of particles
    rho=0.5         # Density

    # Monte Carlo paramters
    dmax=0.1        # maximal distance in one displacement
    ndispl=50       # number of displacement attempts in one move
    nequil=10000    # number of equilibration cycles
    lmax=100000     # number of production cycles
    fsamp=10        # frequency of sampling (number of sampling lmax/nasmp)

    # Parameters of the LJ potential
    eps=1           # L.J. energy          
    sig=1           # L.J. distance 
    #mass=1         # Mass 
    rcut=5.0        # Cutoff radius

    # Temperature
    temp=2.0
    beta=1/temp

    nbins=30
    # Parameters section end
    
    box=(npart/rho)**(1.0/3)        # length of the cubic lattice
    vol=npart*1.0/rho               # volume of the box
    rcut=min(box/2.0,rcut)          # Check that cutoff is not larger than the box 
    r2cut=rcut**2                   # Squared cutoff raidus
    enl=[]                          # list of energy
    prl=[]                          # list of pression
    new_pos=np.array([0.0,0.0,0.0]) # Array for the trial position

    # Set a fixed seed (for debug !!)
    #random.seed(15)
    
    # Energy and pressure correction
    etail= npart*cor_en(rcut, rho, sig, eps)
    ptail= cor_pr(rcut,rho, sig, eps)
    
    # Create initial position
    pos=ini_pos(npart, box)
    print ("Initialization on a lattice: particles placed on a lattice")

    # Timing
    start_time = time.time()

# --------------------------------------------------------------
    print ('Equilibration cycles')
    
    # adjust maximum displacement to get ~50% acceptance rate
    frac=0.5
    lfrac5=[]
    for i in range (nequil) :
      naccpt=0
      if math.fmod (i+1,5)==0 :
        # update dmax after every 5 cycles
        dold=dmax
        dmax=dold*(frac/0.5)
        # limit the change in dmax
        if dmax/dold > 1.5 :
          dmax=dold*1.5
        if dmax/dold < 0.5 :
          dmax=dold*0.5
        if dmax > box/2.0 :
          dmax=box/2.0
    
      frac5=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, False) 
      frac5=frac5[0]
      lfrac5.append(frac5)
      frac=sum(lfrac5)/len(lfrac5) 
      if math.fmod (i+1,200)==0 :
        print ('cycle', i+1, 'frac',round(frac,4),'dmax',round(dmax,4) )
    
    # Write the position of the atoms in a file
    dump_pos('sample_mc.dat', pos, npart, box, dmax)

# --------------------------------------------------------------
 
    print ('Production cycles')

    # Read the position of the atoms in a file
    pos, npart, box, dmax=read_pos('sample_mc.dat')
    
    for i in range (lmax) :
     a1=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, False)  
     if math.fmod(i+1,fsamp)==0:
        a1=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, True)
        enl.append(a1[1])
        prl.append(a1[2])

    enl=np.array(enl)
    prl=np.array(prl)

    # Add tail correction
    enl+=etail
    prl+=ptail
    

    # Write the energy and pressure to a file
    g3=open('energy','w')
    g4=open('pressure','w')
    
    for i in range(len(prl)):
        g3.write (str(enl[i]/(npart))+'\n')
        g4.write (str(prl[i])+'\n')
    
    g3.close()
    g4.close()
    print("--- %s seconds ---" % (time.time() - start_time))


if __name__=="__main__":
    main()
