#!/usr/bin/python3

# The code performs a blocking analysis on the input data
# Output: sigma(I) and tau as a function of the block transformation 
#         step.
# Author: Igor Reshetnyak 2017
#         Sai Lyu, EPFL, Mar 2021
#         Arnaud Lorin 2022

import math
import sys

def main():

    ## paramters
    if len(sys.argv) > 1:
        fn=sys.argv[1]
    else:
        fn='pressure'  # file name
    ntrans=10 # number of applied transformations 
    ## paramters
    
    f1=open(fn,'r')
    ff1=f1.readlines()
    f1.close()
    nline=len(ff1)
    
    if 2**ntrans > nline:
        print('Not enough data for all iterations, reduce ntrans (max {}) or increase number of data point'.format(int(math.log(nline)/math.log(2))))
        raise Exception('Not enough data !')
    g1=open(str(fn)+'_BLOCKING.dat','w')
    g1.write('# Step Sigma_I(M) SD{Sigma_I}       tau(M) SD{tau(M)}\n')
    
    # Create the 0th blocking iteration
    A0list=[]
    for i in range (nline):
      aa=ff1[i].split()[0]
      A0list.append(float(aa))
    
    print("average = {}".format(sum(A0list)/len(A0list)))
    sigma2A=var2(A0list)
    A0len=len(A0list)

    taum,sdtaum,sigmaI,sdsigmaI=compute_error(0, A0list, sigma2A, A0len)
    g1.write(' {:5d} {:10.7f}  {:10.7f}  {:10.7f}  {:10.7f} \n'.format(0,sigmaI,sdsigmaI, taum, sdtaum))

    #print("average = {}".format(sum(A0list)/A0len))
    
    Alist=A0list
    # Iterates the blocking
    for iblock in range (1,ntrans+1): 
      Alist=blocking(Alist)
      # Compute the error
      taum,sdtaum,sigmaI,sdsigmaI=compute_error(iblock, Alist, sigma2A, A0len)
      g1.write(' {:5d} {:10.7f}  {:10.7f}  {:10.7f}  {:10.7f} \n'.format(iblock,sigmaI,sdsigmaI, taum, sdtaum))
      
    g1.close() 

# Blocking iteration, average two elements
def blocking (Alist):
    Blist=[]
    if int(len(Alist)/2)==0:
        raise Exception('Not enough data for the next iteration, reduce ntrans or increase number of data point')
    for i in range (int(len(Alist)/2)) :
        b=0.5*(Alist[2*i]+Alist[2*i+1])
        Blist.append(b)
    return Blist

# Compute the variance of Alist
def var2 (Alist):
    var2=0
    suma=sum(Alist)
    for i in range (len(Alist))  :
        var2 += (Alist[i]-suma/len(Alist))**2/len(Alist)
    return var2  

def compute_error(iblock, Blist, sigma2A, A0len):
    taum=var2(Blist)*2**(iblock-1)/sigma2A     # Blocked correlation time 
    sigmaI=math.sqrt(2*taum*sigma2A/A0len) # Blocked error of the sum
    
    sdtaum=math.sqrt(2.0/len(Blist))*taum        # Error of the blocked correlation time 
    sdsigmaI=math.sqrt(0.5/len(Blist))*sigmaI    # Error of the blocked error of the sum
    return taum,sdtaum,sigmaI,sdsigmaI

if __name__=='__main__':
    main()
